data "aws_iam_policy_document" "lambda-ec2-vpc" {
  statement {
    actions = [
      "ec2:CreateNetworkInterface",
      "ec2:DescribeNetworkInterfaces",
      "ec2:DeleteNetworkInterface",
    ]

    resources = ["*"]

    effect = "Allow"
  }
}

data "aws_iam_policy_document" "lambda-assume-role" {
  statement {
    actions = [
      "sts:AssumeRole",
    ]

    principals = {
      type = "Service"

      identifiers = [
        "lambda.amazonaws.com",
      ]
    }

    effect = "Allow"
  }
}

data "aws_iam_policy_document" "lambda-logs-permission" {
  statement {
    actions = [
      "logs:CreateLogGroup",
      "logs:CreateLogStream",
      "logs:PutLogEvents",
    ]

    resources = [
      "arn:aws:logs:*:*:*",
    ]

    effect = "Allow"
  }
}

data "aws_iam_policy_document" "lambda-invoke-function" {
  statement {
    actions = [
      "lambda:InvokeFunction",
    ]

    resources = [
      "*",
    ]

    effect = "Allow"
  }
}

data "aws_iam_policy_document" "aux" {
  statement {
    actions = [
      "sns:Subscribe",
    ]

    resources = [
      "${aws_sns_topic.sandstorm.arn}",
    ]

    effect = "Allow"
  }

  statement {
    actions = [
      "sqs:CreateQueue",
      "sqs:GetQueueAttributes",
      "sqs:DeleteMessage",
      "sqs:ReceiveMessage",
      "sqs:SetQueueAttributes",
    ]

    resources = [
      "arn:aws:sqs:us-west-2:${var.account_id}:sandstorm*",
    ]

    effect = "Allow"
  }

  statement {
    actions = [
      "kms:Decrypt",
    ]

    resources = [
      "${aws_kms_key.sandstorm.arn}",
    ]

    effect = "Allow"
  }

  statement {
    actions = [
      "iam:GetUser",
      "iam:ListGroupsForUser",
    ]

    resources = [
      "arn:aws:iam::${var.account_id}:user/$${aws:username}",
    ]

    effect = "Allow"
  }

  statement {
    sid = "DynamoDbDenyCrossEnvAccess"

    actions = [
      "dynamodb:UpdateItem",
      "dynamodb:PutItem",
    ]

    resources = [
      "${aws_dynamodb_table.sandstorm.arn}",
    ]

    condition {
      test     = "ForAllValues:StringEquals"
      variable = "dynamodb:Attributes"
      values   = ["cross_env"]
    }

    effect = "Deny"
  }

  statement {
    sid = "AssumeInventoryRole"

    actions = [
      "sts:AssumeRole",
    ]

    resources = [
      "${var.inventory_execute_api_role_arn}",
    ]
  }

  statement {
    sid = "AssumeCloudwatchRole"

    actions = [
      "sts:AssumeRole",
    ]

    resources = [
      "${aws_iam_role.cloudwatch-metrics.arn}",
    ]
  }
}

data "aws_iam_policy_document" "aux-rw" {
  statement {
    sid = "snsSubscribe"

    actions = [
      "sns:Subscribe",
    ]

    resources = [
      "${aws_sns_topic.sandstorm.arn}",
    ]

    effect = "Allow"
  }

  statement {
    sid = "sqsPermissions"

    actions = [
      "sqs:CreateQueue",
      "sqs:GetQueueAttributes",
      "sqs:DeleteMessage",
      "sqs:ReceiveMessage",
      "sqs:SetQueueAttributes",
    ]

    resources = [
      "arn:aws:sqs:us-west-2:${var.account_id}:sandstorm*",
    ]

    effect = "Allow"
  }

  statement {
    sid = "kmsEncryptAndDecrypt"

    actions = [
      "kms:Decrypt",
      "kms:Encrypt",
      "kms:GenerateDataKey",
    ]

    resources = [
      "${aws_kms_key.sandstorm.arn}",
    ]

    effect = "Allow"
  }

  statement {
    sid = "iamGetUser"

    actions = [
      "iam:GetUser",
      "iam:ListGroupsForUser",
    ]

    resources = [
      "arn:aws:iam::${var.account_id}:user/$${aws:username}",
    ]

    effect = "Allow"
  }

  statement {
    sid = "DenyDynamoDbCrossEnvAccess"

    actions = [
      "dynamodb:UpdateItem",
      "dynamodb:PutItem",
    ]

    resources = [
      "${aws_dynamodb_table.sandstorm.arn}",
    ]

    condition {
      test     = "ForAllValues:StringEquals"
      variable = "dynamodb:Attributes"
      values   = ["cross_env"]
    }

    effect = "Deny"
  }

  statement {
    sid = "AssumeInventoryRole"

    actions = [
      "sts:AssumeRole",
    ]

    resources = [
      "${var.inventory_execute_api_role_arn}",
    ]
  }

  statement {
    sid = "AssumeCloudwatchRole"

    actions = [
      "sts:AssumeRole",
    ]

    resources = [
      "${aws_iam_role.cloudwatch-metrics.arn}",
    ]
  }
}

data "aws_iam_policy_document" "sandstorm-assume-role" {
  statement {
    actions = [
      "sts:AssumeRole",
    ]

    principals {
      type = "AWS"

      identifiers = "${var.apiserver_role_allowed_arns}"
    }

    effect = "Allow"
  }
}

data "aws_iam_policy_document" "apiserver" {
  statement {
    sid = "kmsEncryptAndDecrypt"

    actions = [
      "kms:Decrypt",
      "kms:Encrypt",
      "kms:GenerateDataKey",
    ]

    resources = [
      "${aws_kms_key.sandstorm.arn}",
    ]

    effect = "Allow"
  }

  statement {
    sid = "iamRolePolicies"

    actions = [
      "iam:AttachRolePolicy",
      "iam:CreatePolicy",
      "iam:CreatePolicyVersion",
      "iam:CreateRole",
      "iam:DeletePolicy",
      "iam:DeletePolicyVersion",
      "iam:DeleteRole",
      "iam:DetachRolePolicy",
      "iam:GetPolicy",
      "iam:GetPolicyVersion",
      "iam:GetRole",
      "iam:ListAttachedRolePolicies",
      "iam:ListPolicies",
      "iam:ListPolicyVersions",
      "iam:ListRoles",
      "iam:SetDefaultPolicyVersion",
      "iam:UpdateAssumeRolePolicy",
    ]

    resources = [
      "arn:aws:iam::${var.account_id}:policy/sandstorm/${var.secret_env}/templated/policy/",
      "arn:aws:iam::${var.account_id}:policy/sandstorm/${var.secret_env}/templated/policy/*",
      "arn:aws:iam::${var.account_id}:policy/sandstorm-*",
      "arn:aws:iam::${var.account_id}:policy/Sandstorm-*",
      "arn:aws:iam::${var.account_id}:role/sandstorm/${var.secret_env}/templated/role/",
      "arn:aws:iam::${var.account_id}:role/sandstorm/${var.secret_env}/templated/role/*",
      "arn:aws:iam::${var.account_id}:role/sandstorm-*",
      "arn:aws:iam::${var.account_id}:role/Sandstorm-*",
      "arn:aws:iam::${var.account_id}:policy/sandstorm/${var.secret_env}/aux_policy/",
      "arn:aws:iam::${var.account_id}:policy/sandstorm/${var.secret_env}/aux_policy/*",
    ]

    effect = "Allow"
  }

  statement {
    sid = "iamUserGroupsReadWrite"

    actions = [
      "iam:AttachRolePolicy",
      "iam:DetachRolePolicy",
      "iam:GetUser",
      "iam:GetGroup",
      "iam:CreateGroup",
      "iam:AttachGroupPolicy",
      "iam:DetachGroupPolicy",
      "iam:DeleteGroup",
      "iam:AddUserToGroup",
      "iam:RemoveUserFromGroup",
    ]

    resources = [
      "arn:aws:iam::${var.account_id}:policy/*",
      "arn:aws:iam::${var.account_id}:role/*",
      "arn:aws:iam::${var.account_id}:user/sandstorm/${var.secret_env}/sandstorm-*",
      "arn:aws:iam::${var.account_id}:user/sandstorm-*",
      "arn:aws:iam::${var.account_id}:user/Sandstorm-*",
      "arn:aws:iam::${var.account_id}:group/sandstorm/${var.secret_env}",
      "arn:aws:iam::${var.account_id}:group/sandstorm/${var.secret_env}/*",
      "arn:aws:iam::${var.account_id}:group/Sandstorm-*",
      "arn:aws:iam::${var.account_id}:group/sandstorm-*",
      "arn:aws:iam::${var.account_id}:group/sandstorm*",
    ]

    effect = "Allow"
  }

  statement {
    sid = "iamUserGroupRead"

    actions = [
      "iam:ListUsers",
      "iam:ListGroupsForUser",
      "iam:ListGroups",
      "iam:ListAttachedGroupPolicies",
      "iam:ListAttachedRolePolicies",
    ]

    resources = [
      "arn:aws:iam::${var.account_id}:policy/*",
      "arn:aws:iam::${var.account_id}:user/*",
      "arn:aws:iam::${var.account_id}:group/sandstorm/${var.secret_env}",
      "arn:aws:iam::${var.account_id}:group/sandstorm/${var.secret_env}/*",
      "arn:aws:iam::${var.account_id}:role/*",
    ]

    effect = "Allow"
  }

  statement {
    sid = "dynamoDBReadWrite"

    actions = [
      "dynamodb:DeleteItem",
      "dynamodb:PutItem",
      "dynamodb:Query",
      "dynamodb:Scan",
      "dynamodb:DescribeTable",
      "dynamodb:UpdateItem",
      "dynamodb:GetItem",
      "dynamodb:BatchGetItem",
    ]

    resources = [
      "${aws_dynamodb_table.sandstorm.arn}",
      "${aws_dynamodb_table.sandstorm-audit.arn}",
      "${aws_dynamodb_table.namespaces.arn}",
      "${aws_dynamodb_table.role-owners.arn}",
      "${aws_dynamodb_table.sandstorm.arn}/index/namespace_name",
      "${aws_dynamodb_table.sandstorm-audit.arn}/index/namespace_name",
    ]

    effect = "Allow"
  }

  statement {
    sid = "AssumeCloudwatchRole"

    actions = [
      "sts:AssumeRole",
    ]

    resources = [
      "${aws_iam_role.cloudwatch-metrics.arn}",
    ]
  }
}

data "aws_iam_policy_document" "malachai-service-lambda-arp" {
  statement {
    actions = [
      "sts:AssumeRole",
    ]

    principals = {
      type = "AWS"

      identifiers = [
        "${var.malachai_service_lambda_role_arn}",
      ]
    }

    effect = "Allow"
  }
}
