data "aws_iam_policy_document" "sandstorm-dynamodb-stream" {
  statement {
    actions = [
      "dynamodb:GetRecords",
      "dynamodb:GetShardIterator",
      "dynamodb:DescribeStream",
      "dynamodb:ListStreams",
    ]

    resources = [
      "${aws_dynamodb_table.sandstorm.stream_arn}",
    ]

    effect = "Allow"
  }
}

data "aws_iam_policy_document" "sns-admin" {
  statement {
    actions = [
      "sns:*",
    ]

    effect = "Allow"

    resources = [
      "${aws_sns_topic.sandstorm.arn}",
    ]
  }
}

resource "aws_iam_role" "sandstorm-lambda-permissions" {
  name               = "sandstorm-lambda-permissions-${var.env}"
  assume_role_policy = "${data.aws_iam_policy_document.lambda-assume-role.json}"
}

resource "aws_iam_policy" "sandstorm-stream" {
  name        = "sandstorm-lambda-stream-${var.env}"
  path        = "/"
  description = "provides access to sandstorm-${var.env} stream"
  policy      = "${data.aws_iam_policy_document.sandstorm-dynamodb-stream.json}"
}

resource "aws_iam_policy" "sandstorm-sns" {
  name        = "sandstorm-lambda-sns-admin-${var.env}"
  path        = "/"
  description = "provides admin access to sandstorm-${var.env} sns topic"
  policy      = "${data.aws_iam_policy_document.sns-admin.json}"
}

resource "aws_iam_policy" "sandstorm-lambda-invoke-function" {
  name        = "sandstorm-lambda-invoke-function-${var.env}"
  path        = "/"
  description = "provides invoke function permission"
  policy      = "${data.aws_iam_policy_document.lambda-invoke-function.json}"
}

resource "aws_iam_policy_attachment" "sandstorm-stream" {
  name       = "sandstorm-lambda-stream-permission-attachement-${var.env}"
  roles      = ["${aws_iam_role.sandstorm-lambda-permissions.name}"]
  policy_arn = "${aws_iam_policy.sandstorm-stream.arn}"
}

resource "aws_iam_policy_attachment" "sandstorm-sns" {
  name       = "sandstorm-sns-admin-permission-attachement-${var.env}"
  roles      = ["${aws_iam_role.sandstorm-lambda-permissions.name}"]
  policy_arn = "${aws_iam_policy.sandstorm-sns.arn}"
}

resource "aws_iam_policy_attachment" "sandstorm-invoke-function" {
  name       = "sandstorm-lambda-invoke-function-${var.env}"
  roles      = ["${aws_iam_role.sandstorm-lambda-permissions.name}"]
  policy_arn = "${aws_iam_policy.sandstorm-lambda-invoke-function.arn}"
}

resource "aws_lambda_function" "sandstorm-lambda" {
  function_name = "sandstorm-${var.env}-lambda"
  role          = "${aws_iam_role.sandstorm-lambda-permissions.arn}"
  handler       = "DynamoLambda.handler"
  s3_bucket     = "${var.lambda_s3_bucket}"
  s3_key        = "sandstorm-lambda/${local.lambda_source_s3_key}.zip"
  runtime       = "nodejs4.3"
  runtime       = "nodejs4.3"
  timeout       = "300"
  description   = "An Amazon DynamoDB trigger that logs the updates made to a table."
  publish       = true
  depends_on    = ["aws_iam_role.sandstorm-lambda-permissions"]

  environment {
    variables = {
      SANDSTORM_ACCOUNT_ID = "${var.account_id}"
    }
  }
}

resource "aws_lambda_event_source_mapping" "sandstorm-lambda-stream-mapping" {
  batch_size        = 1
  event_source_arn  = "${aws_dynamodb_table.sandstorm.stream_arn}"
  enabled           = true
  function_name     = "${aws_lambda_function.sandstorm-lambda.arn}"
  starting_position = "LATEST"
}
