variable "apiserver_role_name" {}
variable "aux_rw_policy_arn" {}

variable "trusted_account_ids" {
  type = "list"
}

variable "region" {}
variable "account_id" {}
variable "aws_profile" {}
variable "env" {}
variable "sandstorm_table" {}
variable "sandstorm_audit_table" {}
variable "sandstorm_namespace_table" {}

output "agent_role_arn" {
  value = "${aws_iam_role.agent.arn}"
}

provider "aws" {
  version             = "~> 1.0"
  region              = "${var.region}"
  profile             = "${var.aws_profile}"
  allowed_account_ids = ["${var.account_id}"]
}

data "aws_iam_policy_document" "testing-only" {
  statement {
    actions = [
      "sqs:DeleteQueue",
      "sqs:PurgeQueue",
    ]

    effect = "Allow"

    resources = [
      "arn:aws:sqs:us-west-2:${var.account_id}:sandstorm*",
    ]
  }

  statement {
    actions = [
      "cloudwatch:GetMetricStatistics",
    ]

    effect = "Allow"

    resources = ["*"]
  }
}

resource "aws_iam_policy_attachment" "agent-aux-rw" {
  name = "sandstorm-apiserver-agent-rw-attachement-${var.env}"

  roles = [
    "${var.apiserver_role_name}",
    "${aws_iam_role.agent.name}",
  ]

  policy_arn = "${var.aux_rw_policy_arn}"
}

resource "aws_iam_policy" "testing-only" {
  name        = "sandstorm-apiserver-testing-only-${var.env}"
  path        = "/${var.env}/"
  description = "permissions required to clean up resources, query metrics during testing"
  policy      = "${data.aws_iam_policy_document.testing-only.json}"
}

resource "aws_iam_policy_attachment" "agent-aux" {
  name = "sandstorm-testing-only-attachement-${var.env}"

  roles = [
    "${var.apiserver_role_name}",
    "${aws_iam_role.agent.name}",
  ]

  policy_arn = "${aws_iam_policy.testing-only.arn}"
}

data "aws_iam_policy_document" "agent-arp" {
  statement {
    actions = [
      "sts:AssumeRole",
    ]

    principals {
      type = "AWS"

      identifiers = "${formatlist(
        "arn:aws:iam::%s:root",
        var.trusted_account_ids,
      )}"
    }

    effect = "Allow"
  }
}

resource "aws_iam_role" "agent" {
  name               = "sandstorm-agent-${var.env}-rw"
  assume_role_policy = "${data.aws_iam_policy_document.agent-arp.json}"
}

data "aws_iam_policy_document" "agent" {
  statement {
    actions = [
      "dynamodb:DeleteItem",
      "dynamodb:PutItem",
      "dynamodb:Query",
      "dynamodb:Scan",
      "dynamodb:UpdateItem",
      "dynamodb:DescribeTable",
    ]

    effect = "Allow"

    resources = [
      "${var.sandstorm_table}",
      "${var.sandstorm_audit_table}",
      "${var.sandstorm_namespace_table}",
      "${var.sandstorm_table}/index/namespace_name",
      "${var.sandstorm_audit_table}/index/namespace_name",
    ]
  }
}

resource "aws_iam_policy" "agent" {
  name        = "sandstorm-agent-${var.env}"
  path        = "/${var.env}/"
  description = "permission required to run agent testing"
  policy      = "${data.aws_iam_policy_document.agent.json}"
}

resource "aws_iam_policy_attachment" "agent" {
  name       = "sandstorm-agent-attachement-${var.env}"
  roles      = ["${aws_iam_role.agent.name}"]
  policy_arn = "${aws_iam_policy.agent.arn}"
}
