package main

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"net"
	"net/http"
	"os"
	"strconv"
	"time"

	"github.com/afex/hystrix-go/hystrix"
	metricCollector "github.com/afex/hystrix-go/hystrix/metric_collector"
	"github.com/afex/hystrix-go/plugins"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/secretsmanager"
	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/common/config"
	"code.justin.tv/common/gometrics"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/foundation/twitchserver"
	"code.justin.tv/vod/vodapi/internal/auth"
	"code.justin.tv/vod/vodapi/internal/backend"
	"code.justin.tv/vod/vodapi/internal/server"
	"code.justin.tv/vod/vodapi/internal/tracker"
	"code.justin.tv/vod/vodapi/pkg/errorlogger"
	"code.justin.tv/vod/vodapi/pkg/hystrixHelper"
	"code.justin.tv/vod/vodapi/pkg/spade"
	"code.justin.tv/vod/vodapi/pkg/vinyl"
)

const (
	// this file contains config options deployed by sandstorm
	hystrixMetricsSampleRate = 1.0
)

func main() {
	// Load environment, since config doesn't set it until config.Parse().
	env := os.Getenv("ENVIRONMENT")
	if env == "" {
		env = config.Environment()
	}
	log.Printf("Running with environment: %s", env)

	defaultConfig, err := loadJSONConfig("configs/vodapi.json", env)
	if err != nil {
		log.Fatal(err)
	}

	config.Register(defaultConfig)

	err = config.Parse()
	if err != nil {
		log.Fatal(err)
	}

	statter := config.ClusterStatsd()
	gometrics.Monitor(statter, time.Second*5)

	// Fetch secrets from AWS: the Rollbar token.
	secrets, err := configureSecretsManager()
	if err != nil {
		log.Fatal(err)
	}
	output, err := secrets.GetSecretValue(&secretsmanager.GetSecretValueInput{
		SecretId: aws.String("vodapi-rollbar-token"),
	})
	if err != nil {
		log.Fatal(err)
	}
	rollbarToken := *output.SecretString

	errorLogger := configureErrorLogger(rollbarToken, statter)

	vinylClient, err := configureVinyl()
	if err != nil {
		log.Fatal(err)
	}

	configureStatter()

	configureHystrix()

	backend, err := configureBackend(errorLogger, statter, vinylClient)
	if err != nil {
		log.Fatal(err)
	}

	auth, err := configureAuthHandler()
	if err != nil {
		log.Fatal(err)
	}

	server := server.NewServer(auth, backend, errorLogger, statter)

	twitchserver.AddDefaultSignalHandlers()
	err = twitchserver.ListenAndServe(server, twitchserver.NewConfig())
	if err != nil {
		log.Fatal(err)
	}
}

func configureBackend(
	errorLogger errorlogger.ErrorLogger,
	statter statsd.Statter,
	vinylClient vinyl.Client) (backend.Backend, error) {
	tracker, err := configureTracker()
	if err != nil {
		return nil, err
	}

	return backend.NewBackend(tracker, statter, errorLogger, vinylClient), nil
}

func configureVinyl() (vinyl.Client, error) {
	vinylHost := config.Resolve("vinyl-host")
	vinylMaxConcurrents := configResolveInt("vinyl-max-concurrent-requests")
	vinylTimeout := configResolveInt("vinyl-timeout-ms")
	vinylRetryCount := configResolveInt("vinyl-retry-count")
	vinylRetryBackoff := configResolveInt("vinyl-retry-backoff-ms")
	conf := twitchclient.ClientConf{
		Host: vinylHost,
		RoundTripperWrappers: []func(http.RoundTripper) http.RoundTripper{
			hystrixhelper.NewHystrixRoundTripWrapper("vinyl", vinylMaxConcurrents, vinylTimeout, vinylRetryCount, vinylRetryBackoff),
		},
		Stats: config.Statsd(),
		Transport: twitchclient.TransportConf{
			MaxIdleConnsPerHost: configResolveInt("vinyl-max-concurrent-requests") * 2, // we have 2 circuits using the same connection pool
		},
	}
	return vinyl.NewClient(conf)
}

func configureStatter() {
	hostName, err := os.Hostname()
	if err != nil {
		log.Printf("could not retrieve hostname: %s", err.Error())
		hostName = "unknown"
	}

	client, err := plugins.InitializeStatsdCollector(&plugins.StatsdCollectorConfig{
		StatsdAddr: config.StatsdHostPort(),
		Prefix:     fmt.Sprintf("vodapi.%s.%s.hystrix", config.Environment(), hostName),
		SampleRate: hystrixMetricsSampleRate,
		FlushBytes: plugins.LANStatsdFlushBytes,
	})
	if err != nil {
		log.Printf("Couldn't initialize statsd client: %s\n", err.Error())
	}

	metricCollector.Registry.Register(client.NewStatsdCollector)
}

func configureSecretsManager() (*secretsmanager.SecretsManager, error) {
	// This is a bad solution, but previous `config.Resolve("aws-region")` was failing.
	sess, err := session.NewSession(&aws.Config{
		Region: aws.String("us-west-2"),
	})
	if err != nil {
		return nil, err
	}

	return secretsmanager.New(sess), nil
}

func configureErrorLogger(token string, statter statsd.Statter) errorlogger.ErrorLogger {
	env := config.Environment()
	return errorlogger.NewErrorLogger(token, env, statter)
}

func configureHystrix() {
	hystrix.Configure(map[string]hystrix.CommandConfig{
		"spade_track": hystrix.CommandConfig{
			MaxConcurrentRequests: configResolveInt("spade-max-concurrent-requests"),
			Timeout:               configResolveInt("spade-timeout-track-ms"),
			ErrorPercentThreshold: 101, // spade is called asynchronously; disable circuit breaker
		},
	})
}

func configureAuthHandler() (auth.Handler, error) {
	return auth.NewJWTHandler()
}

func configureTracker() (tracker.Tracker, error) {
	spadeHost := config.Resolve("spade-host")

	conf := twitchclient.ClientConf{
		Host: spadeHost,
		Transport: twitchclient.TransportConf{
			MaxIdleConnsPerHost: configResolveInt("spade-max-concurrent-requests"),
		},
	}

	// Spade doesn't run locally.
	if spadeHost == "http://0.0.0.0:0" {
		return &tracker.NoopTracker{}, nil
	}

	spadeClient, err := spade.NewClient(conf)
	if err != nil {
		return nil, err
	}

	return tracker.NewTracker(spadeClient)
}

func configResolveInt(field string) int {
	i, err := strconv.Atoi(config.Resolve(field))
	if err != nil {
		log.Fatal(fmt.Sprintf("config parameter '%s' is not valid: ", field), err)
	}
	return i
}

func customHTTPClient(maxIdleConnsPerHost int) *http.Client {
	return &http.Client{
		Transport: &http.Transport{
			Proxy:               http.ProxyFromEnvironment,
			MaxIdleConnsPerHost: maxIdleConnsPerHost,
			Dial: (&net.Dialer{
				Timeout:   30 * time.Second,
				KeepAlive: 30 * time.Second,
			}).Dial,
		},
	}
}

func loadJSONConfig(confFileName string, env string) (map[string]string, error) {
	result := map[string]string{}
	conf := map[string]map[string]string{}

	jsonFile, err := os.Open(confFileName)
	if err != nil {
		return result, err
	}

	byteValue, _ := ioutil.ReadAll(jsonFile)
	err = jsonFile.Close()
	if err != nil {
		return result, err
	}

	err = json.Unmarshal(byteValue, &conf)
	if err != nil {
		return result, err
	}

	if defaults, ok := conf["default"]; ok {
		result = defaults
	}

	if envSpecific, ok := conf[env]; ok {
		for k, v := range envSpecific {
			result[k] = v
		}
	}

	return result, nil
}
