package backend

import (
	"context"
	"strings"

	"github.com/afex/hystrix-go/hystrix"
	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/vod/vodapi/internal/errors"
	"code.justin.tv/vod/vodapi/internal/models"
	"code.justin.tv/vod/vodapi/internal/tracker"
	"code.justin.tv/vod/vodapi/pkg/errorlogger"
	"code.justin.tv/vod/vodapi/pkg/vinyl"
)

// Backend represents all actions in this layer.
type Backend interface {
	CheckConnection() error

	// vod create
	CreateUpload(ctx context.Context, request *models.CreateUploadRequest) (*models.Vod, error)
	CreateArchive(ctx context.Context, request *models.CreateArchiveRequest) (*models.Vod, error)
	InternalCreateHighlight(ctx context.Context, request *models.InternalCreateHighlightRequest) (*models.Vod, error)
	ManagerCreateHighlight(ctx context.Context, request *models.ManagerCreateHighlightRequest) (*models.Vod, error)

	// single vod request
	PublicGetVodByID(ctx context.Context, request *models.PublicGetVodByIDRequest) (*models.Vod, error)
	InternalGetVodByID(ctx context.Context, request *models.InternalGetVodByIDRequest) (*models.Vod, error)
	ManagerInternalGetVodByID(ctx context.Context, request *models.ManagerInternalGetVodByIDRequest) (*models.Vod, error)
	ManagerGetVodMuteInfo(ctx context.Context, request *models.ManagerGetVodMuteInfoRequest) (*models.VodMuteInfoResponse, error)
	InternalGetVodMuteInfo(ctx context.Context, request *models.InternalGetVodMuteInfoRequest) (*models.VodMuteInfoResponse, error)
	ManagerGetVodMuteInfos(ctx context.Context, request *models.ManagerGetVodMuteInfosRequest) ([]*models.VodMuteInfoResponse, error)
	GetVodByIDIncludeBannedUsers(ctx context.Context, request *models.GetVodByIDIncludeBannedUsersRequest) (*models.Vod, error)
	ManagerGetVodByID(ctx context.Context, request *models.ManagerGetVodByIDRequest) (*models.Vod, error)

	// multiple vod request
	ManagerSearchVods(ctx context.Context, request *models.ManagerSearchVodsRequest) (*models.VodsResponse, error)
	PublicGetVodsByUser(ctx context.Context, request *models.PublicGetVodsByUserRequest) (*models.VodsResponse, error)
	ManagerInternalGetVodsByUser(ctx context.Context, request *models.ManagerInternalGetVodsByUserRequest) (*models.VodsResponse, error)
	ManagerGetVodsByUserIncludeBannedUsers(ctx context.Context, request *models.ManagerGetVodsByUserIncludeBannedUsersRequest) (*models.VodsResponse, error)
	InternalGetVodsByBroadcastIDs(ctx context.Context, request *models.InternalGetVodsByBroadcastIDsRequest) (*models.VodsResponse, error)
	PublicGetVodsByIDs(ctx context.Context, request *models.PublicGetVodsByIDsRequest) (*models.VodsResponse, error)
	InternalGetVodsByIDs(ctx context.Context, request *models.InternalGetVodsByIDsRequest) (*models.VodsResponse, error)
	GetVodsByIDsIncludeBannedUsers(ctx context.Context, request *models.GetVodsByIDsIncludeBannedUsersRequest) (*models.VodsResponse, error)
	PublicGetVodsByUsers(ctx context.Context, request *models.PublicGetVodsByUsersRequest) (*models.VodsResponse, error)
	GetTopVods(ctx context.Context, request *models.GetTopVodsRequest) (*models.VodsResponse, error)

	// update vod
	ManagerUpdateVod(ctx context.Context, request *models.ManagerUpdateVodRequest) (*models.Vod, error)

	// vod delete
	UndeleteVods(ctx context.Context, request *models.UndeleteVodsRequest) (*models.VodsResponse, error)
	SoftDeleteVods(ctx context.Context, request *models.SoftDeleteVodsRequest) (*models.VodsResponse, error)
	InternalVodRemoveRecords(ctx context.Context, request *models.InternalVodRemoveRecordsRequest) (*models.VodsResponse, error)
	HardDeleteVods(ctx context.Context, request *models.HardDeleteVodsRequest) (*models.VodsResponse, error)
	SoftDeleteVodsInInterval(ctx context.Context, request *models.SoftDeleteVodsInIntervalRequest) error
	ManagerSoftDeleteVods(ctx context.Context, request *models.ManagerSoftDeleteVodsRequest) (*models.VodsResponse, error)

	// vod actions
	YoutubeExport(ctx context.Context, request *models.YoutubeExportRequest) error
	FinalizeUpload(ctx context.Context, request *models.FinalizeUploadRequest) error
	UpdateManifest(ctx context.Context, request *models.UpdateManifestRequest) error
	SetViewcounts(ctx context.Context, request *models.SetViewcountsRequest) error

	// vod metadata information
	GetPublicVodAggregationsByIDs(ctx context.Context, request *models.GetPublicVodAggregationsByIDsRequest) (*models.PublicVodAggregationsByIDsResponse, error)
	GetVodPopularity(ctx context.Context, request *models.GetVodPopularityRequest) (*models.VodPopularityResponse, error)
	CreateError(ctx context.Context, request *models.CreateErrorRequest) (*models.CreateErrorResponse, error)

	// vod thumbnails
	CreateThumbnails(ctx context.Context, request *models.CreateThumbnailsRequest) (*models.CreateThumbnailsResponse, error)
	ManagerCreateCustomThumbnailUploadRequest(ctx context.Context, request *models.ManagerCreateCustomThumbnailUploadRequestRequest) (*models.CreateCustomThumbnailUploadRequestResponse, error)
	ManagerDeleteThumbnails(ctx context.Context, request *models.ManagerDeleteThumbnailsRequest) error
	DeleteThumbnails(ctx context.Context, request *models.DeleteThumbnailsRequest) error

	// user properties related to vod
	ManagerGetUserVodProperties(ctx context.Context, request *models.ManagerGetUserVodPropertiesRequest) (*models.UserVodPropertiesResponse, error)
	ManagerUpdateUserVodProperties(ctx context.Context, request *models.ManagerUpdateUserVodPropertiesRequest) (*models.UserVodPropertiesResponse, error)
	ManagerGetUserVideoPrivacyProperties(ctx context.Context, request *models.ManagerGetUserVideoPrivacyPropertiesRequest) (*models.UserVideoPrivacyPropertiesResponse, error)
	ManagerUpdateUserVideoPrivacyProperties(ctx context.Context, request *models.ManagerUpdateUserVideoPrivacyPropertiesRequest) (*models.UserVideoPrivacyPropertiesResponse, error)

	// vod appeals
	ManagerCreateVodAppeals(ctx context.Context, request *models.ManagerCreateVodAppealsRequest) error
	GetVodAppeals(ctx context.Context, request *models.GetVodAppealsRequest) (*models.VodAppealsResponse, error)
	ResolveTrackAppeal(ctx context.Context, request *models.ResolveTrackAppealRequest) error
	ResolveVodAppeal(ctx context.Context, request *models.ResolveVodAppealRequest) error
	CreateAudibleMagicResponses(ctx context.Context, request *models.CreateAudibleMagicResponsesRequest) (*models.AudibleMagicResponsesResponse, error)
	GetAudibleMagicResponses(ctx context.Context, request *models.GetAudibleMagicResponsesRequest) (*models.AudibleMagicResponsesResponse, error)
	UpdateAudibleMagicResponses(ctx context.Context, request *models.UpdateAudibleMagicResponsesRequest) (*models.AudibleMagicResponsesResponse, error)
}

type backendImpl struct {
	errorlogger errorlogger.ErrorLogger
	stats       statsd.Statter
	tracker     tracker.Tracker
	vinyl       vinyl.Client
}

// NewBackend creates a new backend instance
func NewBackend(
	tracker tracker.Tracker,
	stats statsd.Statter,
	errorlogger errorlogger.ErrorLogger,
	vinylClient vinyl.Client,
) Backend {
	return &backendImpl{
		errorlogger: errorlogger,
		stats:       stats,
		tracker:     tracker,
		vinyl:       vinylClient,
	}
}

func (b *backendImpl) ErrorHandle(err error) error {
	if err == nil {
		return nil
	}

	if twitchClientError, ok := err.(*twitchclient.Error); ok {
		if twitchClientError.StatusCode >= 500 {
			b.errorlogger.LogErrorWithExtras("Vinyl backend error", "vinyl_backend_error", map[string]interface{}{
				"error": err.Error(),
			})
		}
	} else {
		if isHystrixError(err) {
			b.errorlogger.LogErrorWithExtras("Hystrix backend error", "hystrix_backend_error", map[string]interface{}{
				"error": err.Error(),
			})
		} else {
			b.errorlogger.LogErrorWithExtras("General backend error", "general_backend_error", map[string]interface{}{
				"error": err.Error(),
			})
		}
	}
	return &errors.BackendError{
		Err: err,
	}
}

func isHystrixError(err error) bool {
	errStr := err.Error()
	isCircuitOpen := strings.Contains(errStr, hystrix.ErrCircuitOpen.Error())
	isTimeout := strings.Contains(errStr, hystrix.ErrTimeout.Error())
	isMaxConcurrency := strings.Contains(errStr, hystrix.ErrMaxConcurrency.Error())
	return isCircuitOpen || isTimeout || isMaxConcurrency
}
