package converters

import (
	"strconv"
	"testing"
	"time"

	vinyl "code.justin.tv/vod/vinyl/client"
	"code.justin.tv/vod/vodapi/internal/models"
	vinylWrapper "code.justin.tv/vod/vodapi/pkg/vinyl"
	"code.justin.tv/vod/vodapi/rpc/vodapi/utils"
	. "github.com/smartystreets/goconvey/convey"
)

func TestAppealsAmrs(t *testing.T) {
	Convey("when using appeals and amrs convertesr", t, func() {
		utils.StubTimeNow()

		Convey("when converting internal to vinyl create appeals", func() {
			input := &models.ManagerCreateVodAppealsRequest{
				VodID:          1234,
				City:           "city",
				Country:        "country",
				FullName:       "full_name",
				State:          "state",
				StreetAddress1: "street_address1",
				StreetAddress2: "street_address2",
				Zipcode:        "zipcode",
				UserID:         2345,
				TrackAppeals: []models.TrackAppealRequest{
					models.TrackAppealRequest{
						AudibleMagicResponseID: 1234,
						Reason:                 "reason",
					},
				},
			}
			output := vinyl.CreateAppealsInput{
				VodID: strconv.FormatInt(input.VodID, 10),
				VodAppeal: vinyl.VodAppeal{
					City:           input.City,
					Country:        input.Country,
					FullName:       input.FullName,
					State:          input.State,
					StreetAddress1: input.StreetAddress1,
					StreetAddress2: input.StreetAddress2,
					Zipcode:        input.Zipcode,
				},
				TrackAppeals: []vinyl.TrackAppeal{
					vinyl.TrackAppeal{
						AudibleMagicResponseID: input.TrackAppeals[0].AudibleMagicResponseID,
						Reason:                 input.TrackAppeals[0].Reason,
					},
				},
			}
			So(output, ShouldResemble, ConvertInternalToVinylCreateAppeals(input))
		})
		Convey("when converting vinyl appeals response", func() {
			input := &vinylWrapper.GetAppealsResponse{
				TotalCount: 100,
				Appeals: []vinylWrapper.Appeal{
					vinylWrapper.Appeal{
						ID:             1234,
						Priority:       true,
						VodID:          2345,
						City:           "city",
						Country:        "country",
						FullName:       "full_name",
						State:          "state",
						StreetAddress1: "street_address1",
						StreetAddress2: "street_address2",
						Zipcode:        "zipcode",
						CreatedAt:      time.Now(),
						UpdatedAt:      time.Now(),
					},
				},
			}
			output := &models.VodAppealsResponse{
				TotalCount: 100,
				Appeals: []models.VodAppeal{
					models.VodAppeal{
						VodAppealID:    input.Appeals[0].ID,
						Priority:       input.Appeals[0].Priority,
						VodID:          input.Appeals[0].VodID,
						City:           input.Appeals[0].City,
						Country:        input.Appeals[0].Country,
						FullName:       input.Appeals[0].FullName,
						State:          input.Appeals[0].State,
						StreetAddress1: input.Appeals[0].StreetAddress1,
						StreetAddress2: input.Appeals[0].StreetAddress2,
						Zipcode:        input.Appeals[0].Zipcode,
						CreatedAt:      input.Appeals[0].CreatedAt,
						UpdatedAt:      input.Appeals[0].UpdatedAt,
					},
				},
			}
			So(output, ShouldResemble, ConvertVinylAppealsResponse(input))
		})
		Convey("when converting internal to vinyl create audible magic response", func() {
			input := models.CreateAudibleMagicResponseRequest{
				VodID:                1234,
				Title:                "title",
				Performer:            "performer",
				Genre:                "genre",
				Artist:               "artist",
				AlbumTitle:           "album_title",
				Song:                 "song",
				Isrc:                 "isrc",
				IsMatch:              true,
				MatchOffsetSeconds:   10,
				MatchDurationSeconds: 20,
				ScanOffsetSeconds:    30,
				ScanDurationSeconds:  40,
				MuteOffsetSeconds:    50,
				MuteDurationSeconds:  60,
				AudibleMagicItemID:   "item_id",
			}
			output := vinyl.CreateAMRInput{
				VodID:                input.VodID,
				Title:                input.Title,
				Performer:            input.Performer,
				Genre:                input.Genre,
				Artist:               input.Artist,
				AlbumTitle:           input.AlbumTitle,
				Song:                 input.Song,
				Isrc:                 input.Isrc,
				MatchOffsetSeconds:   input.MatchOffsetSeconds,
				MatchDurationSeconds: input.MatchDurationSeconds,
				ScanOffsetSeconds:    input.ScanOffsetSeconds,
				ScanDurationSeconds:  input.ScanDurationSeconds,
				MuteOffsetSeconds:    input.MuteOffsetSeconds,
				MuteDurationSeconds:  input.MuteDurationSeconds,
				AmItemID:             input.AudibleMagicItemID,
			}
			So(output, ShouldResemble, ConvertInternalToVinylCreateAudibleMagicResponse(input))
		})
		Convey("when converting to internal update amr", func() {
			input := &models.UpdateAudibleMagicResponsesRequest{
				UpdateAudibleMagicResponses: []models.UpdateAudibleMagicResponseRequest{
					models.UpdateAudibleMagicResponseRequest{
						AudibleMagicResponseID: 9999,
						Title:                utils.StrPtr("title"),
						Performer:            utils.StrPtr("performer"),
						Genre:                utils.StrPtr("genre"),
						Artist:               utils.StrPtr("artist"),
						AlbumTitle:           utils.StrPtr("album_title"),
						Song:                 utils.StrPtr("song"),
						Isrc:                 utils.StrPtr("isrc"),
						IsMatch:              true,
						MatchOffsetSeconds:   utils.Int64Ptr(10),
						MatchDurationSeconds: utils.Int64Ptr(20),
						ScanOffsetSeconds:    utils.Int64Ptr(30),
						ScanDurationSeconds:  utils.Int64Ptr(40),
						MuteOffsetSeconds:    utils.Int64Ptr(50),
						MuteDurationSeconds:  utils.Int64Ptr(60),
						AudibleMagicItemID:   utils.StrPtr("item_id"),
					},
				},
			}
			output := &vinylWrapper.UpdateAudibleMagicResponsesInput{
				UpdateAudibleMagicResponses: []vinylWrapper.UpdateAudibleMagicResponse{
					vinylWrapper.UpdateAudibleMagicResponse{
						AudibleMagicResponseID: input.UpdateAudibleMagicResponses[0].AudibleMagicResponseID,
						Title:                input.UpdateAudibleMagicResponses[0].Title,
						Performer:            input.UpdateAudibleMagicResponses[0].Performer,
						Genre:                input.UpdateAudibleMagicResponses[0].Genre,
						Artist:               input.UpdateAudibleMagicResponses[0].Artist,
						AlbumTitle:           input.UpdateAudibleMagicResponses[0].AlbumTitle,
						Song:                 input.UpdateAudibleMagicResponses[0].Song,
						Isrc:                 input.UpdateAudibleMagicResponses[0].Isrc,
						IsMatch:              input.UpdateAudibleMagicResponses[0].IsMatch,
						MatchOffsetSeconds:   input.UpdateAudibleMagicResponses[0].MatchOffsetSeconds,
						MatchDurationSeconds: input.UpdateAudibleMagicResponses[0].MatchDurationSeconds,
						ScanOffsetSeconds:    input.UpdateAudibleMagicResponses[0].ScanOffsetSeconds,
						ScanDurationSeconds:  input.UpdateAudibleMagicResponses[0].ScanDurationSeconds,
						MuteOffsetSeconds:    input.UpdateAudibleMagicResponses[0].MuteOffsetSeconds,
						MuteDurationSeconds:  input.UpdateAudibleMagicResponses[0].MuteDurationSeconds,
						AmItemID:             input.UpdateAudibleMagicResponses[0].AudibleMagicItemID,
					},
				},
			}
			So(output, ShouldResemble, ConvertInternalToVinylUpdateAudibleMagicResponse(input))
		})
		Convey("when converting vinyl audible magic responses", func() {
			input := []*vinyl.AMR{
				&vinyl.AMR{
					VodID:                1234,
					Title:                utils.StrPtr("title"),
					Performer:            utils.StrPtr("performer"),
					Genre:                utils.StrPtr("genre"),
					Artist:               utils.StrPtr("artist"),
					AlbumTitle:           utils.StrPtr("album_title"),
					Song:                 utils.StrPtr("song"),
					Isrc:                 utils.StrPtr("isrc"),
					IsMatch:              true,
					MatchOffsetSeconds:   utils.Int64Ptr(10),
					MatchDurationSeconds: utils.Int64Ptr(20),
					ScanOffsetSeconds:    utils.Int64Ptr(30),
					ScanDurationSeconds:  utils.Int64Ptr(40),
					MuteOffsetSeconds:    utils.Int64Ptr(50),
					MuteDurationSeconds:  utils.Int64Ptr(60),
					AmItemID:             utils.StrPtr("item_id"),
				},
			}
			output := &models.AudibleMagicResponsesResponse{
				AudibleMagicResponses: []models.AudibleMagicResponse{
					models.AudibleMagicResponse{
						VodID:                input[0].VodID,
						Title:                *input[0].Title,
						Performer:            *input[0].Performer,
						Genre:                *input[0].Genre,
						Artist:               *input[0].Artist,
						AlbumTitle:           *input[0].AlbumTitle,
						Song:                 *input[0].Song,
						Isrc:                 *input[0].Isrc,
						IsMatch:              input[0].IsMatch,
						MatchOffsetSeconds:   input[0].MatchOffsetSeconds,
						MatchDurationSeconds: input[0].MatchDurationSeconds,
						ScanOffsetSeconds:    input[0].ScanOffsetSeconds,
						ScanDurationSeconds:  input[0].ScanDurationSeconds,
						MuteOffsetSeconds:    input[0].MuteOffsetSeconds,
						MuteDurationSeconds:  input[0].MuteDurationSeconds,
						AudibleMagicItemID:   *input[0].AmItemID,
					},
				},
			}
			So(output, ShouldResemble, ConvertVinylAudibleMagicResponses(input))
		})
	})
}
