package converters

import (
	"strings"

	vinyl "code.justin.tv/vod/vinyl/client"
	"code.justin.tv/vod/vodapi/internal/models"
	vinylclient "code.justin.tv/vod/vodapi/pkg/vinyl"
	"code.justin.tv/vod/vodapi/rpc/vodapi/utils"
)

// ConvertVinylVod converts Vod from vinyl to internal model
func ConvertVinylVod(vinylVod *vinyl.Vod) *models.Vod {
	var failure *models.CreateErrorResponse
	if vinylVod.Failure != nil {
		failure = &models.CreateErrorResponse{
			VodID:        vinylVod.Failure.VodID,
			ErrorCode:    vinylVod.Failure.ErrorCode,
			ErrorMessage: vinylVod.Failure.ErrorMessage,
			CreatedAt:    vinylVod.Failure.CreatedAt,
		}
	}
	return &models.Vod{
		BroadcastID:         vinylVod.BroadcastID,
		BroadcastType:       vinylVod.BroadcastType,
		BroadcasterSoftware: vinylVod.BroadcasterSoftware,
		Communities:         vinylVod.Communities,
		CreatedAt:           vinylVod.CreatedAt,
		CreatedBy:           vinylVod.CreatedBy,
		DeleteAt:            vinylVod.DeleteAt,
		Deleted:             vinylVod.Deleted,
		Description:         vinylVod.Description,
		DescriptionHTML:     vinylVod.DescriptionHTML,
		Duration:            int64(vinylVod.Duration),
		Fps:                 vinylVod.Fps,
		Game:                vinylVod.Game,
		ID:                  vinylVod.ID,
		Language:            vinylVod.Language,
		Manifest:            vinylVod.Manifest,
		Offset:              int64(vinylVod.Offset),
		Origin:              vinylVod.Origin,
		OwnerID:             vinylVod.OwnerID,
		Resolutions:         vinylVod.Resolutions,
		SourceArchiveID:     vinylVod.SourceArchiveID,
		StartedOn:           vinylVod.StartedOn,
		Status:              vinylVod.Status,
		TagList:             strings.Split(vinylVod.TagList, ","),
		Title:               vinylVod.Title,
		TotalLength:         int64(vinylVod.TotalLength),
		UpdatedAt:           vinylVod.UpdatedAt,
		URI:                 vinylVod.URI,
		Viewable:            utils.ConvertStrPtrToStr(vinylVod.Viewable),
		ViewableAt:          vinylVod.ViewableAt,
		Views:               vinylVod.Views,
		PublishedAt:         vinylVod.PublishedAt,
		IncrementViewURL:    vinylVod.IncrementViewURL,
		Qualities:           vinylVod.Qualities,
		PreviewTemplate:     vinylVod.PreviewTemplate,
		ShowFormats:         ConvertVinylShowFormats(vinylVod.ShowFormats),
		ThumbnailTemplates:  ConvertVinylThumbnailTemplates(vinylVod.ThumbnailTemplates),
		Thumbnails:          nil,
		SeekPreviewsURL:     vinylVod.SeekPreviewsURL,
		AnimatedPreviewURL:  vinylVod.AnimatedPreviewURL,
		Path:                vinylVod.Path,
		URL:                 vinylVod.URL,
		HandwaveLatencyMS:   vinylVod.HandwaveLatencyMS,
		Failure:             failure,
	}
}

// CreateArchiveVodRequest converts create archive request
func CreateArchiveVodRequest(request *models.CreateArchiveRequest) vinylclient.CreateArchiveVod {
	return vinylclient.CreateArchiveVod{
		BroadcastID:         request.BroadcastID,
		BroadcastType:       "archive",
		BroadcasterSoftware: &request.BroadcasterSoftware,
		Status:              "recording",
		Title:               request.Title,
		OwnerID:             request.ChannelID,
		Offset:              0,
		Duration:            int(request.Duration),
		URI:                 request.URI,
		Origin:              utils.StrPtr("s3"),
		StartedOn:           utils.FromTimePtr(request.RecordedOn),
		Manifest:            &request.Manifest,
		Deleted:             &request.Deleted,
		HandwaveLatencyMS:   &request.HandwaveLatencyMS,
	}
}

// InternalCreateHighlightRequest converts an internal create highlight
func InternalCreateHighlightRequest(request *models.InternalCreateHighlightRequest) vinylclient.InternalCreateHighlightVod {
	showFormatsPtr := ConvertToVinylShowFormats(request.ShowFormats)
	var showFormats map[string]map[string]interface{}
	if showFormatsPtr != nil {
		showFormats = *showFormatsPtr
	}
	return vinylclient.InternalCreateHighlightVod{
		SourceArchiveID: request.SourceArchiveID,
		BroadcastID:     request.BroadcastID,
		BroadcastType:   "highlight",
		Status:          "unprocessed",
		Title:           request.Title,
		OwnerID:         utils.StrToInt64(request.OwnerID),
		Duration:        request.Duration,
		URI:             request.URI,
		ShowFormats:     showFormats,
	}
}

// ConvertVinylVods converts Vods from vinyl to internal model
func ConvertVinylVods(vinylVods []*vinyl.Vod) *models.VodsResponse {
	vods := make([]*models.Vod, len(vinylVods))
	for i, vod := range vinylVods {
		vods[i] = ConvertVinylVod(vod)
	}
	return &models.VodsResponse{
		Vods: vods,
	}
}

// ConvertVodAggregations converts vod aggregations
func ConvertVodAggregations(vodAggregations *vinyl.VodsAggregation) *models.PublicVodAggregationsByIDsResponse {
	return &models.PublicVodAggregationsByIDsResponse{
		TotalCount:    int64(vodAggregations.TotalCount),
		TotalDuration: int64(vodAggregations.TotalDuration),
	}
}

// ConvertVodPopularity converts vod popularity
func ConvertVodPopularity(vodPopularity *vinyl.VodPopularityResponse) *models.VodPopularityResponse {
	return &models.VodPopularityResponse{
		Popular: vodPopularity.Popular,
	}
}

// ConvertCreateError converts create error
func ConvertCreateError(createError *vinyl.CreateErrorResponse) *models.CreateErrorResponse {
	return &models.CreateErrorResponse{
		VodID:        createError.VodID,
		ErrorCode:    createError.ErrorCode,
		ErrorMessage: createError.ErrorMessage,
		CreatedAt:    createError.CreatedAt,
	}
}
