package backend

import (
	"context"
	"errors"
	"testing"

	service_errors "code.justin.tv/vod/vodapi/internal/errors"
	"code.justin.tv/vod/vodapi/internal/models"
	"code.justin.tv/vod/vodapi/internal/testdata"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func TestCreateArchive(t *testing.T) {
	Convey("when calling CreateArchive", t, func() {
		internalArchive := testdata.ArchiveVod()
		createArchive := &models.CreateArchiveRequest{
			BroadcastID:         internalArchive.BroadcastID,
			BroadcasterSoftware: "broadcaster",
			RecordedOn:          nil,
			Title:               "title",
			ChannelID:           internalArchive.OwnerID,
			Duration:            internalArchive.Duration,
			URI:                 internalArchive.URI,
			Manifest:            "manifest",
			Deleted:             false,
			HandwaveLatencyMS:   500,
		}

		backend, mocks := initTestableBackend()

		Convey("when CreateArchive succeeds", func() {
			archive := testdata.VinylArchiveVod()
			mocks.vinyl.On("CreateArchive", mock.Anything, mock.Anything, mock.Anything).Return(archive, nil)
			vod, err := backend.CreateArchive(context.Background(), createArchive)
			So(err, ShouldBeNil)
			So(internalArchive, ShouldResemble, vod)
		})
		Convey("when CreateArchive fails", func() {
			mocks.MockErrorHandle()
			vinylErr := errors.New("something bad happened")
			mocks.vinyl.On("CreateArchive", mock.Anything, mock.Anything, mock.Anything).Return(nil, vinylErr)
			_, err := backend.CreateArchive(context.Background(), createArchive)
			So(err, ShouldResemble, &service_errors.BackendError{Err: vinylErr})
		})
		mocks.AssertExpectations(t)
	})
}
