package backend

import (
	"context"
	"errors"
	"testing"

	service_errors "code.justin.tv/vod/vodapi/internal/errors"
	"code.justin.tv/vod/vodapi/internal/models"
	"code.justin.tv/vod/vodapi/internal/testdata"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func TestCreateAudibleMagicResponses(t *testing.T) {
	Convey("when calling CreateAudibleMagicResponses", t, func() {
		backend, mocks := initTestableBackend()
		ctx := context.Background()

		request := &models.CreateAudibleMagicResponsesRequest{
			AudibleMagicResponses: []models.CreateAudibleMagicResponseRequest{
				models.CreateAudibleMagicResponseRequest{
					VodID:                testdata.VodID1,
					Title:                "title",
					Performer:            "performer",
					Genre:                "genre",
					Artist:               "artist",
					AlbumTitle:           "album_title",
					Song:                 "song",
					Isrc:                 "isrc",
					MatchOffsetSeconds:   10,
					MatchDurationSeconds: 20,
					ScanOffsetSeconds:    30,
					ScanDurationSeconds:  40,
					MuteOffsetSeconds:    50,
					MuteDurationSeconds:  60,
					AudibleMagicItemID:   "amr_id",
				},
			},
		}

		Convey("when CreateAudibleMagicResponses succeeds", func() {
			audibleMagicResponses := testdata.VinylAudibleMagicResponses()
			mocks.vinyl.On("CreateAMR", mock.Anything, mock.Anything, mock.Anything).Return(audibleMagicResponses[0], nil)
			thumbnailsResp, err := backend.CreateAudibleMagicResponses(ctx, request)
			So(err, ShouldBeNil)
			So(thumbnailsResp, ShouldNotBeNil)
		})
		Convey("when CreateAudibleMagicResponses fails", func() {
			mocks.MockErrorHandle()
			vinylErr := errors.New("something bad happened")
			mocks.vinyl.On("CreateAMR", mock.Anything, mock.Anything, mock.Anything).Return(nil, vinylErr)
			_, err := backend.CreateAudibleMagicResponses(ctx, request)
			So(err, ShouldResemble, &service_errors.BackendError{Err: vinylErr})
		})
		mocks.AssertExpectations(t)
	})
}
