package backend

import (
	"errors"
	"strings"
	"testing"

	vinyl "code.justin.tv/vod/vinyl/client"
	service_errors "code.justin.tv/vod/vodapi/internal/errors"
	"code.justin.tv/vod/vodapi/internal/models"
	"code.justin.tv/vod/vodapi/internal/testdata"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
	"golang.org/x/net/context"
)

type testcase struct {
	desc             string
	createRequest    models.CreateUploadRequest
	resposneVinylVod *vinyl.Vod
	responseErr      error
	expectedVod      *models.Vod
	expectedErr      error
}

func TestCreateUpload(t *testing.T) {
	Convey("when calling CreateUpload", t, func() {
		upload := testdata.VinylUploadVod()
		internalUpload := testdata.UploadVod()
		createUpload := &models.CreateUploadRequest{
			BroadcastType: upload.BroadcastType,
			BroadcastID:   upload.BroadcastID,
			OwnerID:       upload.OwnerID,
			Description:   "",
			Duration:      int64(upload.Duration),
			URI:           upload.URI,
			Manifest:      *upload.Manifest,
			Game:          *upload.Game,
			Communities:   upload.Communities,
			Language:      *upload.Language,
			TagList:       strings.Split(upload.TagList, ","),
			Viewable:      *upload.Viewable,
		}

		backend, mocks := initTestableBackend()

		Convey("when CreateUpload succeeds", func() {
			mocks.vinyl.On("CreateUpload", mock.Anything, mock.Anything, mock.Anything).Return(upload, nil)
			vod, err := backend.CreateUpload(context.Background(), createUpload)
			So(err, ShouldBeNil)
			So(internalUpload, ShouldResemble, vod)
		})
		Convey("when CreateUpload fails", func() {
			mocks.MockErrorHandle()
			vinylErr := errors.New("something bad happened")
			mocks.vinyl.On("CreateUpload", mock.Anything, mock.Anything, mock.Anything).Return(nil, vinylErr)
			_, err := backend.CreateUpload(context.Background(), createUpload)
			So(err, ShouldResemble, &service_errors.BackendError{Err: vinylErr})
		})
		mocks.AssertExpectations(t)
	})
}
