package backend

import (
	"context"
	"errors"
	"testing"

	vinyl "code.justin.tv/vod/vinyl/client"
	service_errors "code.justin.tv/vod/vodapi/internal/errors"
	"code.justin.tv/vod/vodapi/internal/models"
	"code.justin.tv/vod/vodapi/internal/testdata"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func TestManagerGetVodsByUserIncludeBannedUsers(t *testing.T) {
	Convey("when calling ManagerGetVodsByUserIncludeBannedUsers", t, func() {
		backend, mocks := initTestableBackend()
		ctx := context.Background()

		request := &models.ManagerGetVodsByUserIncludeBannedUsersRequest{
			ChannelID: 1234,
			Language:  "english",
			Status:    "status",
			Sort:      "sort",
			Limit:     100,
			Offset:    0,
			UserID:    1234,
		}

		Convey("when ManagerGetVodsByUserIncludeBannedUsers succeeds", func() {
			vinylUpload := testdata.VinylUploadVod()
			internalUpload := testdata.UploadVod()
			vinylVods := []*vinyl.Vod{vinylUpload}
			mocks.vinyl.On("GetVodsByUserIncludeBannedUsers", ctx, mock.Anything, mock.Anything, mock.Anything).Return(vinylVods, nil)
			vods, err := backend.ManagerGetVodsByUserIncludeBannedUsers(ctx, request)
			So(err, ShouldBeNil)
			So(len(vods.Vods), ShouldEqual, 1)
			So(vods.Vods[0], ShouldResemble, internalUpload)
		})
		Convey("when ManagerGetVodsByUserIncludeBannedUsers with broadcast type succeeds", func() {
			requestWithBroadcastType := &models.ManagerGetVodsByUserIncludeBannedUsersRequest{
				ChannelID:      1234,
				BroadcastTypes: []string{"archive"},
				Language:       "english",
				Status:         "status",
				Sort:           "sort",
				Limit:          100,
				Offset:         0,
				UserID:         1234,
			}

			vinylUpload := testdata.VinylUploadVod()
			internalUpload := testdata.UploadVod()
			vinylVods := []*vinyl.Vod{vinylUpload}
			mocks.vinyl.On("GetVodsByUserIncludeBannedUsers", ctx, mock.Anything, mock.Anything, mock.Anything).Return(vinylVods, nil)
			vods, err := backend.ManagerGetVodsByUserIncludeBannedUsers(ctx, requestWithBroadcastType)
			So(err, ShouldBeNil)
			So(len(vods.Vods), ShouldEqual, 1)
			So(vods.Vods[0], ShouldResemble, internalUpload)
		})
		Convey("when ManagerGetVodsByUserIncludeBannedUsers fails", func() {
			mocks.MockErrorHandle()
			vinylErr := errors.New("something bad happened")
			mocks.vinyl.On("GetVodsByUserIncludeBannedUsers", ctx, mock.Anything, mock.Anything, mock.Anything).Return(nil, vinylErr)
			_, err := backend.ManagerGetVodsByUserIncludeBannedUsers(ctx, request)
			So(err, ShouldResemble, &service_errors.BackendError{Err: vinylErr})
		})
		mocks.AssertExpectations(t)
	})
}
