package errors

import (
	"fmt"
	"net/http"
)

// WithStatusCode an interface for errors returned by the API.
type WithStatusCode interface {
	StatusCode() int
	Error() string
}

// WithBody is an interface for errors that have a custom JSON body.
type WithBody interface {
	Body() map[string]interface{}
}

// Unauthorized is returned when a user is not authorized to perform an action.
type Unauthorized struct {
	Err error
}

// Error returns the string representation of an Unauthorized.
func (e *Unauthorized) Error() string {
	return fmt.Sprintf("You are not authorized: %s", e.Err.Error())
}

// StatusCode returns the http status code this error maps to.
func (e *Unauthorized) StatusCode() int {
	return http.StatusUnauthorized
}

// GenericInternal is a generic, internal error.
type GenericInternal struct{}

// Error returns the string representation of a GenericInternal.
func (e *GenericInternal) Error() string {
	return fmt.Sprintf("Generic Internal Error")
}

// StatusCode returns the http status code this error maps to.
func (e *GenericInternal) StatusCode() int {
	return http.StatusInternalServerError
}

// BackendError wraps a backend error
type BackendError struct {
	Err error
}

// Error returns the error message
func (e *BackendError) Error() string {
	return e.Err.Error()
}
