package converters

import (
	"testing"
	"time"

	"code.justin.tv/vod/vodapi/internal/models"
	rpc "code.justin.tv/vod/vodapi/rpc/vodapi"
	"code.justin.tv/vod/vodapi/rpc/vodapi/utils"
	"github.com/golang/protobuf/ptypes/wrappers"
	. "github.com/smartystreets/goconvey/convey"
)

func TestAppealsAmrs(t *testing.T) {
	Convey("when using appeals and amrs convertesr", t, func() {
		utils.StubTimeNow()

		Convey("when converting to internal create appeals", func() {
			input := &rpc.ManagerCreateVodAppealsRequest{
				VodId:          "1234",
				City:           "city",
				Country:        "country",
				FullName:       "full_name",
				State:          "state",
				StreetAddress1: "street_address1",
				StreetAddress2: "street_address2",
				Zipcode:        "zipcode",
				UserId:         "2345",
				TrackAppeals: []*rpc.TrackAppealRequest{
					&rpc.TrackAppealRequest{
						AudibleMagicResponseId: "1234",
						Reason:                 "reason",
					},
				},
			}
			output := &models.ManagerCreateVodAppealsRequest{
				VodID:          utils.StrToInt64(input.VodId),
				City:           input.City,
				Country:        input.Country,
				FullName:       input.FullName,
				State:          input.State,
				StreetAddress1: input.StreetAddress1,
				StreetAddress2: input.StreetAddress2,
				Zipcode:        input.Zipcode,
				UserID:         utils.StrToInt64(input.UserId),
				TrackAppeals: []models.TrackAppealRequest{
					models.TrackAppealRequest{
						AudibleMagicResponseID: utils.StrToInt64(input.TrackAppeals[0].AudibleMagicResponseId),
						Reason:                 input.TrackAppeals[0].Reason,
					},
				},
			}
			So(output, ShouldResemble, ConvertToInternalCreateAppeals(input))
		})
		Convey("when converting from internal appeals response", func() {
			input := &models.VodAppealsResponse{
				TotalCount: 1,
				Appeals: []models.VodAppeal{
					models.VodAppeal{
						VodAppealID:    1234,
						Priority:       true,
						VodID:          2345,
						City:           "city",
						Country:        "country",
						FullName:       "full_name",
						State:          "state",
						StreetAddress1: "street_address1",
						StreetAddress2: "street_address2",
						Zipcode:        "zipcode",
						CreatedAt:      time.Now(),
						UpdatedAt:      time.Now(),
					},
				},
			}
			output := &rpc.VodAppealsResponse{
				TotalCount: 1,
				Appeals: []*rpc.VodAppeal{
					&rpc.VodAppeal{
						VodAppealId:    utils.Int64ToStr(input.Appeals[0].VodAppealID),
						Priority:       input.Appeals[0].Priority,
						VodId:          utils.Int64ToStr(input.Appeals[0].VodID),
						City:           input.Appeals[0].City,
						Country:        input.Appeals[0].Country,
						FullName:       input.Appeals[0].FullName,
						State:          input.Appeals[0].State,
						StreetAddress1: input.Appeals[0].StreetAddress1,
						StreetAddress2: input.Appeals[0].StreetAddress2,
						Zipcode:        input.Appeals[0].Zipcode,
						CreatedAt:      utils.ProtobufTimeAsTimestamp(&input.Appeals[0].CreatedAt),
						UpdatedAt:      utils.ProtobufTimeAsTimestamp(&input.Appeals[0].UpdatedAt),
					},
				},
			}
			So(output, ShouldResemble, ConvertFromInternalAppealsResponse(input))
		})
		Convey("when converting to internal create amr", func() {
			input := &rpc.CreateAudibleMagicResponsesRequest{
				AudibleMagicResponses: []*rpc.CreateAudibleMagicResponseRequest{
					&rpc.CreateAudibleMagicResponseRequest{
						VodId:                "1234",
						Title:                "title",
						Performer:            "performer",
						Genre:                "genre",
						Artist:               "artist",
						AlbumTitle:           "album_title",
						Song:                 "song",
						Isrc:                 "isrc",
						IsMatch:              true,
						MatchOffsetSeconds:   10,
						MatchDurationSeconds: 20,
						ScanOffsetSeconds:    30,
						ScanDurationSeconds:  40,
						MuteOffsetSeconds:    50,
						MuteDurationSeconds:  60,
						AudibleMagicItemId:   "item_id",
					},
				},
			}
			output := &models.CreateAudibleMagicResponsesRequest{
				AudibleMagicResponses: []models.CreateAudibleMagicResponseRequest{
					models.CreateAudibleMagicResponseRequest{
						VodID:                utils.StrToInt64(input.AudibleMagicResponses[0].VodId),
						Title:                input.AudibleMagicResponses[0].Title,
						Performer:            input.AudibleMagicResponses[0].Performer,
						Genre:                input.AudibleMagicResponses[0].Genre,
						Artist:               input.AudibleMagicResponses[0].Artist,
						AlbumTitle:           input.AudibleMagicResponses[0].AlbumTitle,
						Song:                 input.AudibleMagicResponses[0].Song,
						Isrc:                 input.AudibleMagicResponses[0].Isrc,
						IsMatch:              input.AudibleMagicResponses[0].IsMatch,
						MatchOffsetSeconds:   input.AudibleMagicResponses[0].MatchOffsetSeconds,
						MatchDurationSeconds: input.AudibleMagicResponses[0].MatchDurationSeconds,
						ScanOffsetSeconds:    input.AudibleMagicResponses[0].ScanOffsetSeconds,
						ScanDurationSeconds:  input.AudibleMagicResponses[0].ScanDurationSeconds,
						MuteOffsetSeconds:    input.AudibleMagicResponses[0].MuteOffsetSeconds,
						MuteDurationSeconds:  input.AudibleMagicResponses[0].MuteDurationSeconds,
						AudibleMagicItemID:   input.AudibleMagicResponses[0].AudibleMagicItemId,
					},
				},
			}
			So(output, ShouldResemble, ConvertToInternalCreateAudibleMagicResponses(input))
		})
		Convey("when converting to internal update amr", func() {
			input := &rpc.UpdateAudibleMagicResponsesRequest{
				UpdateAudibleMagicResponses: []*rpc.UpdateAudibleMagicResponseRequest{
					&rpc.UpdateAudibleMagicResponseRequest{
						AudibleMagicResponseId: "9999",
						Title:                &wrappers.StringValue{Value: "title"},
						Performer:            &wrappers.StringValue{Value: "performer"},
						Genre:                &wrappers.StringValue{Value: "genre"},
						Artist:               &wrappers.StringValue{Value: "artist"},
						AlbumTitle:           &wrappers.StringValue{Value: "album_title"},
						Song:                 &wrappers.StringValue{Value: "song"},
						Isrc:                 &wrappers.StringValue{Value: "isrc"},
						IsMatch:              true,
						MatchOffsetSeconds:   &wrappers.Int64Value{Value: 10},
						MatchDurationSeconds: &wrappers.Int64Value{Value: 20},
						ScanOffsetSeconds:    &wrappers.Int64Value{Value: 30},
						ScanDurationSeconds:  &wrappers.Int64Value{Value: 40},
						MuteOffsetSeconds:    &wrappers.Int64Value{Value: 50},
						MuteDurationSeconds:  &wrappers.Int64Value{Value: 60},
						AudibleMagicItemId:   &wrappers.StringValue{Value: "item_id"},
					},
				},
			}
			output := &models.UpdateAudibleMagicResponsesRequest{
				UpdateAudibleMagicResponses: []models.UpdateAudibleMagicResponseRequest{
					models.UpdateAudibleMagicResponseRequest{
						AudibleMagicResponseID: utils.StrToInt64(input.UpdateAudibleMagicResponses[0].AudibleMagicResponseId),
						Title:                &input.UpdateAudibleMagicResponses[0].Title.Value,
						Performer:            &input.UpdateAudibleMagicResponses[0].Performer.Value,
						Genre:                &input.UpdateAudibleMagicResponses[0].Genre.Value,
						Artist:               &input.UpdateAudibleMagicResponses[0].Artist.Value,
						AlbumTitle:           &input.UpdateAudibleMagicResponses[0].AlbumTitle.Value,
						Song:                 &input.UpdateAudibleMagicResponses[0].Song.Value,
						Isrc:                 &input.UpdateAudibleMagicResponses[0].Isrc.Value,
						IsMatch:              input.UpdateAudibleMagicResponses[0].IsMatch,
						MatchOffsetSeconds:   &input.UpdateAudibleMagicResponses[0].MatchOffsetSeconds.Value,
						MatchDurationSeconds: &input.UpdateAudibleMagicResponses[0].MatchDurationSeconds.Value,
						ScanOffsetSeconds:    &input.UpdateAudibleMagicResponses[0].ScanOffsetSeconds.Value,
						ScanDurationSeconds:  &input.UpdateAudibleMagicResponses[0].ScanDurationSeconds.Value,
						MuteOffsetSeconds:    &input.UpdateAudibleMagicResponses[0].MuteOffsetSeconds.Value,
						MuteDurationSeconds:  &input.UpdateAudibleMagicResponses[0].MuteDurationSeconds.Value,
						AudibleMagicItemID:   &input.UpdateAudibleMagicResponses[0].AudibleMagicItemId.Value,
					},
				},
			}
			So(output, ShouldResemble, ConvertToInternalUpdateAudibleMagicResponses(input))
		})
		Convey("when converting from internal amr", func() {
			matchOffsetSeconds := int64(10)
			matchDurationSeconds := int64(20)
			scanOffsetSeconds := int64(30)
			scanDurationSeconds := int64(40)
			muteOffsetSeconds := int64(50)
			muteDurationSeconds := int64(60)

			input := &models.AudibleMagicResponsesResponse{
				AudibleMagicResponses: []models.AudibleMagicResponse{
					models.AudibleMagicResponse{
						VodID:                1234,
						Title:                "title",
						Performer:            "performer",
						Genre:                "genre",
						Artist:               "artist",
						AlbumTitle:           "album_title",
						Song:                 "song",
						Isrc:                 "isrc",
						IsMatch:              true,
						MatchOffsetSeconds:   &matchOffsetSeconds,
						MatchDurationSeconds: &matchDurationSeconds,
						ScanOffsetSeconds:    &scanOffsetSeconds,
						ScanDurationSeconds:  &scanDurationSeconds,
						MuteOffsetSeconds:    &muteOffsetSeconds,
						MuteDurationSeconds:  &muteDurationSeconds,
						AudibleMagicItemID:   "item_id",
					},
				},
			}
			output := &rpc.AudibleMagicResponsesResponse{
				AudibleMagicResponses: []*rpc.AudibleMagicResponse{
					&rpc.AudibleMagicResponse{
						VodId:                utils.Int64ToStr(input.AudibleMagicResponses[0].VodID),
						Title:                input.AudibleMagicResponses[0].Title,
						Performer:            input.AudibleMagicResponses[0].Performer,
						Genre:                input.AudibleMagicResponses[0].Genre,
						Artist:               input.AudibleMagicResponses[0].Artist,
						AlbumTitle:           input.AudibleMagicResponses[0].AlbumTitle,
						Song:                 input.AudibleMagicResponses[0].Song,
						Isrc:                 input.AudibleMagicResponses[0].Isrc,
						IsMatch:              input.AudibleMagicResponses[0].IsMatch,
						MatchOffsetSeconds:   utils.ProtobufInt64Value(input.AudibleMagicResponses[0].MatchOffsetSeconds),
						MatchDurationSeconds: utils.ProtobufInt64Value(input.AudibleMagicResponses[0].MatchDurationSeconds),
						ScanOffsetSeconds:    utils.ProtobufInt64Value(input.AudibleMagicResponses[0].ScanOffsetSeconds),
						ScanDurationSeconds:  utils.ProtobufInt64Value(input.AudibleMagicResponses[0].ScanDurationSeconds),
						MuteOffsetSeconds:    utils.ProtobufInt64Value(input.AudibleMagicResponses[0].MuteOffsetSeconds),
						MuteDurationSeconds:  utils.ProtobufInt64Value(input.AudibleMagicResponses[0].MuteDurationSeconds),
						AudibleMagicItemId:   input.AudibleMagicResponses[0].AudibleMagicItemID,
					},
				},
			}
			So(output, ShouldResemble, ConvertFromInternalAudibleMagicResponses(input))
		})
	})
}
