package converters

import (
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/vod/vodapi/internal/errors"
	"code.justin.tv/vod/vodapi/pkg/errorlogger"
	"github.com/twitchtv/twirp"
)

const (
	// InvalidTwirpErrorCode is the invalid twirp error code
	InvalidTwirpErrorCode = "invalid"
)

// ConvertToTwirpError converts an error to the appropriate twirp error
func ConvertToTwirpError(err error, errorLogger errorlogger.ErrorLogger) error {
	internalError := err
	backendError, ok := err.(*errors.BackendError)
	if ok {
		internalError = backendError.Err
	} else {
		errorLogger.LogErrorWithExtras("Unwrapped backend error", "unwrapped_backend_error", map[string]interface{}{
			"error": err.Error(),
		})
	}

	twitchClientError, ok := internalError.(*twitchclient.Error)
	var twirpErr twirp.Error
	if !ok {
		twirpErr = twirp.InternalErrorWith(err)
	} else {
		twirpErr = twirp.NewError(toTwirpErrorCode(twitchClientError.StatusCode), twitchClientError.Message)
	}

	return twirpErr
}

func toTwirpErrorCode(statusCode int) twirp.ErrorCode {
	switch statusCode {
	case 408: // RequestTimeout
		return twirp.DeadlineExceeded
	case 500: // Internal Server Error
		return twirp.Internal
	case 400: // BadRequest
		return twirp.InvalidArgument
	case 404: // Not Found
		return twirp.NotFound
	case 409: // Conflict
		return twirp.AlreadyExists
	case 403: // Forbidden
		return twirp.PermissionDenied
	case 401: // Unauthorized
		return twirp.Unauthenticated
	case 412: // Precondition Failed
		return twirp.FailedPrecondition
	case 501: // Not Implemented
		return twirp.Unimplemented
	case 503: // Service Unavailable
		return twirp.Unavailable
	case 200: // OK
		return twirp.NoError
	default:
		return InvalidTwirpErrorCode
	}
}
