package converters_test

import (
	"errors"
	"testing"

	"github.com/stretchr/testify/mock"
	"github.com/twitchtv/twirp"

	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/vod/vodapi/internal/server/converters"

	. "github.com/smartystreets/goconvey/convey"

	errorloggerMock "code.justin.tv/vod/vodapi/pkg/errorlogger/mocks"
)

func TestConvertToTwirpError(t *testing.T) {
	Convey("when calling ConvertToTwirpError", t, func() {
		mockErrorLogger := &errorloggerMock.ErrorLogger{}

		Convey("when is not twitchclient error", func() {
			mockErrorLogger.On("LogErrorWithExtras", mock.Anything, mock.Anything, mock.Anything)
			err := converters.ConvertToTwirpError(errors.New("hello"), mockErrorLogger)
			So(err, ShouldNotBeNil)
			twirpErr, ok := err.(twirp.Error)
			So(ok, ShouldBeTrue)
			So(twirpErr.Code(), ShouldEqual, twirp.Internal)
		})
		Convey("when RequestTimeout", func() {
			mockErrorLogger.On("LogErrorWithExtras", mock.Anything, mock.Anything, mock.Anything)
			err := converters.ConvertToTwirpError(&twitchclient.Error{StatusCode: 408}, mockErrorLogger)
			So(err, ShouldNotBeNil)
			twirpErr, ok := err.(twirp.Error)
			So(ok, ShouldBeTrue)
			So(twirpErr.Code(), ShouldEqual, twirp.DeadlineExceeded)
		})
		Convey("when Internal", func() {
			mockErrorLogger.On("LogErrorWithExtras", mock.Anything, mock.Anything, mock.Anything)
			err := converters.ConvertToTwirpError(&twitchclient.Error{StatusCode: 500}, mockErrorLogger)
			So(err, ShouldNotBeNil)
			twirpErr, ok := err.(twirp.Error)
			So(ok, ShouldBeTrue)
			So(twirpErr.Code(), ShouldEqual, twirp.Internal)
		})
		Convey("when InvalidArgument", func() {
			mockErrorLogger.On("LogErrorWithExtras", mock.Anything, mock.Anything, mock.Anything)
			err := converters.ConvertToTwirpError(&twitchclient.Error{StatusCode: 400}, mockErrorLogger)
			So(err, ShouldNotBeNil)
			twirpErr, ok := err.(twirp.Error)
			So(ok, ShouldBeTrue)
			So(twirpErr.Code(), ShouldEqual, twirp.InvalidArgument)
		})
		Convey("when NotFound", func() {
			mockErrorLogger.On("LogErrorWithExtras", mock.Anything, mock.Anything, mock.Anything)
			err := converters.ConvertToTwirpError(&twitchclient.Error{StatusCode: 404}, mockErrorLogger)
			So(err, ShouldNotBeNil)
			twirpErr, ok := err.(twirp.Error)
			So(ok, ShouldBeTrue)
			So(twirpErr.Code(), ShouldEqual, twirp.NotFound)
		})
		Convey("when AlreadyExists", func() {
			mockErrorLogger.On("LogErrorWithExtras", mock.Anything, mock.Anything, mock.Anything)
			err := converters.ConvertToTwirpError(&twitchclient.Error{StatusCode: 409}, mockErrorLogger)
			So(err, ShouldNotBeNil)
			twirpErr, ok := err.(twirp.Error)
			So(ok, ShouldBeTrue)
			So(twirpErr.Code(), ShouldEqual, twirp.AlreadyExists)
		})
		Convey("when PermissionDenied", func() {
			mockErrorLogger.On("LogErrorWithExtras", mock.Anything, mock.Anything, mock.Anything)
			err := converters.ConvertToTwirpError(&twitchclient.Error{StatusCode: 403}, mockErrorLogger)
			So(err, ShouldNotBeNil)
			twirpErr, ok := err.(twirp.Error)
			So(ok, ShouldBeTrue)
			So(twirpErr.Code(), ShouldEqual, twirp.PermissionDenied)
		})
		Convey("when Unauthenticated", func() {
			mockErrorLogger.On("LogErrorWithExtras", mock.Anything, mock.Anything, mock.Anything)
			err := converters.ConvertToTwirpError(&twitchclient.Error{StatusCode: 401}, mockErrorLogger)
			So(err, ShouldNotBeNil)
			twirpErr, ok := err.(twirp.Error)
			So(ok, ShouldBeTrue)
			So(twirpErr.Code(), ShouldEqual, twirp.Unauthenticated)
		})
		Convey("when FailedPrecondition", func() {
			mockErrorLogger.On("LogErrorWithExtras", mock.Anything, mock.Anything, mock.Anything)
			err := converters.ConvertToTwirpError(&twitchclient.Error{StatusCode: 412}, mockErrorLogger)
			So(err, ShouldNotBeNil)
			twirpErr, ok := err.(twirp.Error)
			So(ok, ShouldBeTrue)
			So(twirpErr.Code(), ShouldEqual, twirp.FailedPrecondition)
		})
		Convey("when Unimplemented", func() {
			mockErrorLogger.On("LogErrorWithExtras", mock.Anything, mock.Anything, mock.Anything)
			err := converters.ConvertToTwirpError(&twitchclient.Error{StatusCode: 501}, mockErrorLogger)
			So(err, ShouldNotBeNil)
			twirpErr, ok := err.(twirp.Error)
			So(ok, ShouldBeTrue)
			So(twirpErr.Code(), ShouldEqual, twirp.Unimplemented)
		})
		Convey("when Unavailable", func() {
			mockErrorLogger.On("LogErrorWithExtras", mock.Anything, mock.Anything, mock.Anything)
			err := converters.ConvertToTwirpError(&twitchclient.Error{StatusCode: 503}, mockErrorLogger)
			So(err, ShouldNotBeNil)
			twirpErr, ok := err.(twirp.Error)
			So(ok, ShouldBeTrue)
			So(twirpErr.Code(), ShouldEqual, twirp.Unavailable)
		})
		Convey("when NoError", func() {
			mockErrorLogger.On("LogErrorWithExtras", mock.Anything, mock.Anything, mock.Anything)
			err := converters.ConvertToTwirpError(&twitchclient.Error{StatusCode: 200}, mockErrorLogger)
			So(err, ShouldNotBeNil)
			twirpErr, ok := err.(twirp.Error)
			So(ok, ShouldBeTrue)
			So(twirpErr.Code(), ShouldEqual, twirp.NoError)
		})
		Convey("when Unknown", func() {
			mockErrorLogger.On("LogErrorWithExtras", mock.Anything, mock.Anything, mock.Anything)
			err := converters.ConvertToTwirpError(&twitchclient.Error{StatusCode: 9999}, mockErrorLogger)
			So(err, ShouldNotBeNil)
			twirpErr, ok := err.(twirp.Error)
			So(ok, ShouldBeTrue)
			So(twirpErr.Code(), ShouldEqual, twirp.Internal)
		})
	})
}
