package converters

import (
	"code.justin.tv/vod/vodapi/internal/models"
	rpc "code.justin.tv/vod/vodapi/rpc/vodapi"
	"code.justin.tv/vod/vodapi/rpc/vodapi/utils"
)

// ConvertInternalMuteInfo converts MuteInfo from internal representation to rpc
func ConvertInternalMuteInfo(muteInfo *models.VodMuteInfoResponse) *rpc.VodMuteInfoResponse {
	audibleMagicResponses := make([]*rpc.AudibleMagicResponse, len(muteInfo.AudibleMagicResponses))
	for i, audibleMagicResponse := range muteInfo.AudibleMagicResponses {
		audibleMagicResponses[i] = &rpc.AudibleMagicResponse{
			AudibleMagicResponseId: utils.Int64ToStr(audibleMagicResponse.AudibleMagicResponseID),
			VodId:                utils.Int64ToStr(audibleMagicResponse.VodID),
			Title:                audibleMagicResponse.Title,
			Performer:            audibleMagicResponse.Performer,
			Genre:                audibleMagicResponse.Genre,
			Artist:               audibleMagicResponse.Artist,
			AlbumTitle:           audibleMagicResponse.AlbumTitle,
			Song:                 audibleMagicResponse.Song,
			Isrc:                 audibleMagicResponse.Isrc,
			IsMatch:              audibleMagicResponse.IsMatch,
			MatchOffsetSeconds:   utils.ProtobufInt64Value(audibleMagicResponse.MatchOffsetSeconds),
			MatchDurationSeconds: utils.ProtobufInt64Value(audibleMagicResponse.MatchDurationSeconds),
			ScanOffsetSeconds:    utils.ProtobufInt64Value(audibleMagicResponse.ScanOffsetSeconds),
			ScanDurationSeconds:  utils.ProtobufInt64Value(audibleMagicResponse.ScanDurationSeconds),
			MuteOffsetSeconds:    utils.ProtobufInt64Value(audibleMagicResponse.MuteOffsetSeconds),
			MuteDurationSeconds:  utils.ProtobufInt64Value(audibleMagicResponse.MuteDurationSeconds),
			AudibleMagicItemId:   audibleMagicResponse.AudibleMagicItemID,
			CreatedAt:            utils.ProtobufTimeAsTimestamp(&audibleMagicResponse.CreatedAt),
			UnmutedAt:            utils.ProtobufTimeAsTimestamp(audibleMagicResponse.UnmutedAt),
		}
	}

	mutedSegments := make([]*rpc.VodMutedSegment, len(muteInfo.MutedSegments))
	for i, mutedSegment := range muteInfo.MutedSegments {
		mutedSegments[i] = &rpc.VodMutedSegment{
			Offset:   mutedSegment.Offset,
			Duration: mutedSegment.Duration,
		}
	}

	vodAppeal := &rpc.VodAppeal{
		VodAppealId:    utils.Int64ToStr(muteInfo.VodAppeal.VodAppealID),
		ResolvedAt:     utils.ProtobufTimeAsTimestamp(muteInfo.VodAppeal.ResolvedAt),
		CreatedAt:      utils.ProtobufTimeAsTimestamp(&muteInfo.VodAppeal.CreatedAt),
		UpdatedAt:      utils.ProtobufTimeAsTimestamp(&muteInfo.VodAppeal.UpdatedAt),
		VodId:          utils.Int64ToStr(muteInfo.VodAppeal.VodID),
		Priority:       muteInfo.VodAppeal.Priority,
		FullName:       muteInfo.VodAppeal.FullName,
		StreetAddress1: muteInfo.VodAppeal.StreetAddress1,
		StreetAddress2: muteInfo.VodAppeal.StreetAddress2,
		City:           muteInfo.VodAppeal.City,
		State:          muteInfo.VodAppeal.State,
		Zipcode:        muteInfo.VodAppeal.Zipcode,
		Country:        muteInfo.VodAppeal.Country,
	}

	trackAppeals := make([]*rpc.TrackAppeal, len(muteInfo.TrackAppeals))
	for i, trackAppeal := range muteInfo.TrackAppeals {
		trackAppeals[i] = &rpc.TrackAppeal{
			TrackAppealId:          utils.Int64ToStr(trackAppeal.TrackAppealID),
			AudibleMagicResponseId: utils.Int64ToStr(trackAppeal.AudibleMagicResponseID),
			VodAppealId:            utils.Int64ToStr(trackAppeal.VodAppealID),
			Reason:                 trackAppeal.Reason,
			CreatedAt:              utils.ProtobufTimeAsTimestamp(&trackAppeal.CreatedAt),
			UpdatedAt:              utils.ProtobufTimeAsTimestamp(&trackAppeal.UpdatedAt),
			ResolvedAt:             utils.ProtobufTimeAsTimestamp(trackAppeal.ResolvedAt),
		}
	}

	return &rpc.VodMuteInfoResponse{
		Id: utils.Int64ToStr(muteInfo.ID),
		AudibleMagicResponses: audibleMagicResponses,
		MutedSegments:         mutedSegments,
		VodAppeal:             vodAppeal,
		TrackAppeals:          trackAppeals,
		IsMuted:               utils.ProtobufBoolValue(muteInfo.IsMuted),
		CanCreateAppeal:       utils.ProtobufBoolValue(muteInfo.CanCreateAppeal),
		HasPendingAppeal:      utils.ProtobufBoolValue(muteInfo.HasPendingAppeal),
	}
}

// ConvertInternalMuteInfos converts MuteInfos from internal representation to rpc
func ConvertInternalMuteInfos(muteInfos []*models.VodMuteInfoResponse) []*rpc.VodMuteInfoResponse {
	resp := make([]*rpc.VodMuteInfoResponse, len(muteInfos))
	for i, muteInfo := range muteInfos {
		resp[i] = ConvertInternalMuteInfo(muteInfo)
	}
	return resp
}
