package converters

import (
	"testing"
	"time"

	"github.com/golang/protobuf/ptypes/wrappers"

	"code.justin.tv/vod/vodapi/internal/models"
	rpc "code.justin.tv/vod/vodapi/rpc/vodapi"
	"code.justin.tv/vod/vodapi/rpc/vodapi/utils"
	. "github.com/smartystreets/goconvey/convey"
)

func TestMuteInfo(t *testing.T) {
	Convey("when using mute info convertesr", t, func() {
		utils.StubTimeNow()

		Convey("when converting from internal mute info", func() {
			isMuted := true
			canCreateAppeal := true
			hasPendingAppeal := true
			matchOffsetSeconds := int64(10)
			matchDurationSeconds := int64(20)
			scanOffsetSeconds := int64(30)
			scanDurationSeconds := int64(40)
			muteOffsetSeconds := int64(50)
			muteDurationSeconds := int64(60)

			input := &models.VodMuteInfoResponse{
				ID: 8888,
				AudibleMagicResponses: []models.AudibleMagicResponse{
					models.AudibleMagicResponse{
						AudibleMagicResponseID: 2345,
						VodID:                3456,
						Title:                "title",
						Performer:            "performer",
						Genre:                "genre",
						Artist:               "artist",
						AlbumTitle:           "album_title",
						Song:                 "song",
						Isrc:                 "isrc",
						IsMatch:              true,
						MatchOffsetSeconds:   &matchOffsetSeconds,
						MatchDurationSeconds: &matchDurationSeconds,
						ScanOffsetSeconds:    &scanOffsetSeconds,
						ScanDurationSeconds:  &scanDurationSeconds,
						MuteOffsetSeconds:    &muteOffsetSeconds,
						MuteDurationSeconds:  &muteDurationSeconds,
						AudibleMagicItemID:   "item_id",
						CreatedAt:            time.Now(),
						UnmutedAt:            utils.TimePtr(time.Now()),
					},
				},
				MutedSegments: []models.VodMutedSegment{
					models.VodMutedSegment{
						Offset:   10,
						Duration: 20,
					},
				},
				VodAppeal: models.VodAppeal{
					VodAppealID:    1234,
					CreatedAt:      time.Now(),
					UpdatedAt:      time.Now(),
					VodID:          2345,
					Priority:       true,
					FullName:       "full_name",
					StreetAddress1: "street_address1",
					StreetAddress2: "street_address2",
					City:           "city",
					State:          "state",
					Zipcode:        "zipcode",
					Country:        "country",
				},
				TrackAppeals: []models.TrackAppeal{
					models.TrackAppeal{
						TrackAppealID:          7777,
						AudibleMagicResponseID: 8888,
						VodAppealID:            9999,
						Reason:                 "reason",
						CreatedAt:              time.Now(),
						UpdatedAt:              time.Now(),
					},
				},
				IsMuted:          &isMuted,
				CanCreateAppeal:  &canCreateAppeal,
				HasPendingAppeal: &hasPendingAppeal,
			}
			output := &rpc.VodMuteInfoResponse{
				Id: "8888",
				AudibleMagicResponses: []*rpc.AudibleMagicResponse{
					&rpc.AudibleMagicResponse{
						AudibleMagicResponseId: utils.Int64ToStr(input.AudibleMagicResponses[0].AudibleMagicResponseID),
						VodId:                utils.Int64ToStr(input.AudibleMagicResponses[0].VodID),
						Title:                input.AudibleMagicResponses[0].Title,
						Performer:            input.AudibleMagicResponses[0].Performer,
						Genre:                input.AudibleMagicResponses[0].Genre,
						Artist:               input.AudibleMagicResponses[0].Artist,
						AlbumTitle:           input.AudibleMagicResponses[0].AlbumTitle,
						Song:                 input.AudibleMagicResponses[0].Song,
						Isrc:                 input.AudibleMagicResponses[0].Isrc,
						IsMatch:              input.AudibleMagicResponses[0].IsMatch,
						MatchOffsetSeconds:   utils.ProtobufInt64Value(input.AudibleMagicResponses[0].MatchOffsetSeconds),
						MatchDurationSeconds: utils.ProtobufInt64Value(input.AudibleMagicResponses[0].MatchDurationSeconds),
						ScanOffsetSeconds:    utils.ProtobufInt64Value(input.AudibleMagicResponses[0].ScanOffsetSeconds),
						ScanDurationSeconds:  utils.ProtobufInt64Value(input.AudibleMagicResponses[0].ScanDurationSeconds),
						MuteOffsetSeconds:    utils.ProtobufInt64Value(input.AudibleMagicResponses[0].MuteOffsetSeconds),
						MuteDurationSeconds:  utils.ProtobufInt64Value(input.AudibleMagicResponses[0].MuteDurationSeconds),
						AudibleMagicItemId:   input.AudibleMagicResponses[0].AudibleMagicItemID,
						CreatedAt:            utils.ProtobufTimeAsTimestamp(&input.AudibleMagicResponses[0].CreatedAt),
						UnmutedAt:            utils.ProtobufTimeAsTimestamp(input.AudibleMagicResponses[0].UnmutedAt),
					},
				},
				MutedSegments: []*rpc.VodMutedSegment{
					&rpc.VodMutedSegment{
						Offset:   10,
						Duration: 20,
					},
				},
				VodAppeal: &rpc.VodAppeal{
					VodAppealId:    utils.Int64ToStr(input.VodAppeal.VodAppealID),
					CreatedAt:      utils.ProtobufTimeAsTimestamp(&input.VodAppeal.CreatedAt),
					UpdatedAt:      utils.ProtobufTimeAsTimestamp(&input.VodAppeal.UpdatedAt),
					VodId:          utils.Int64ToStr(input.VodAppeal.VodID),
					Priority:       input.VodAppeal.Priority,
					FullName:       input.VodAppeal.FullName,
					StreetAddress1: input.VodAppeal.StreetAddress1,
					StreetAddress2: input.VodAppeal.StreetAddress2,
					City:           input.VodAppeal.City,
					State:          input.VodAppeal.State,
					Zipcode:        input.VodAppeal.Zipcode,
					Country:        input.VodAppeal.Country,
				},
				TrackAppeals: []*rpc.TrackAppeal{
					&rpc.TrackAppeal{
						TrackAppealId:          utils.Int64ToStr(input.TrackAppeals[0].TrackAppealID),
						AudibleMagicResponseId: utils.Int64ToStr(input.TrackAppeals[0].AudibleMagicResponseID),
						VodAppealId:            utils.Int64ToStr(input.TrackAppeals[0].VodAppealID),
						Reason:                 input.TrackAppeals[0].Reason,
						CreatedAt:              utils.ProtobufTimeAsTimestamp(&input.TrackAppeals[0].CreatedAt),
						UpdatedAt:              utils.ProtobufTimeAsTimestamp(&input.TrackAppeals[0].UpdatedAt),
					},
				},
				IsMuted:          &wrappers.BoolValue{Value: isMuted},
				CanCreateAppeal:  &wrappers.BoolValue{Value: canCreateAppeal},
				HasPendingAppeal: &wrappers.BoolValue{Value: hasPendingAppeal},
			}
			So(output, ShouldResemble, ConvertInternalMuteInfo(input))
		})
	})
}
