package converters

import (
	"time"

	"code.justin.tv/vod/vodapi/internal/models"
	rpc "code.justin.tv/vod/vodapi/rpc/vodapi"
	"code.justin.tv/vod/vodapi/rpc/vodapi/utils"
)

// ConvertInternalVod converts Vod from internal representation to rpc
func ConvertInternalVod(vod *models.Vod) *rpc.Vod {
	var failure *rpc.CreateErrorResponse
	if vod.Failure != nil {
		failure = &rpc.CreateErrorResponse{
			VodId:        utils.Int64ToStr(vod.Failure.VodID),
			ErrorCode:    vod.Failure.ErrorCode,
			ErrorMessage: vod.Failure.ErrorMessage,
			CreatedAt:    utils.ProtobufTimeAsTimestamp(&vod.Failure.CreatedAt),
		}
	}
	return &rpc.Vod{
		BroadcastId:         utils.Int64ToStr(vod.BroadcastID),
		BroadcastType:       rpc.ConvertInternalVodType(vod.BroadcastType),
		BroadcasterSoftware: utils.ProtobufStringValue(vod.BroadcasterSoftware),
		Communities:         vod.Communities,
		CreatedAt:           utils.ProtobufTimeAsTimestamp(&vod.CreatedAt),
		CreatedBy:           utils.ProtobufStringValue(utils.Int64PtrToStrPtr(vod.CreatedBy)),
		DeleteAt:            utils.ProtobufTimeAsTimestamp(vod.DeleteAt),
		Deleted:             utils.ProtobufBoolValue(vod.Deleted),
		Description:         utils.ProtobufStringValue(vod.Description),
		DescriptionHtml:     utils.ProtobufStringValue(vod.DescriptionHTML),
		Duration:            vod.Duration,
		Fps:                 vod.Fps,
		Game:                utils.ProtobufStringValue(vod.Game),
		Id:                  utils.Int64ToStr(vod.ID),
		Language:            utils.ProtobufStringValue(vod.Language),
		Manifest:            utils.ProtobufStringValue(vod.Manifest),
		Offset:              vod.Offset,
		Origin:              utils.ProtobufStringValue(vod.Origin),
		OwnerId:             utils.Int64ToStr(vod.OwnerID),
		Resolutions:         vod.Resolutions,
		SourceArchiveId:     utils.ProtobufStringValue(utils.Int64PtrToStrPtr(vod.SourceArchiveID)),
		StartedOn:           utils.ProtobufTimeAsTimestamp(&vod.StartedOn),
		Status:              rpc.ConvertInternalVodStatus(vod.Status),
		TagList:             vod.TagList,
		Title:               utils.ProtobufStringValue(vod.Title),
		TotalLength:         vod.TotalLength,
		UpdatedAt:           utils.ProtobufTimeAsTimestamp(&vod.UpdatedAt),
		Uri:                 vod.URI,
		Viewable:            rpc.ConvertInternalVodViewable(vod.Viewable),
		ViewableAt:          utils.ProtobufTimeAsTimestamp(vod.ViewableAt),
		Views:               vod.Views,
		PublishedAt:         utils.ProtobufTimeAsTimestamp(vod.PublishedAt),
		IncrementViewUrl:    vod.IncrementViewURL,
		Qualities:           vod.Qualities,
		PreviewTemplate:     vod.PreviewTemplate,
		ShowFormats:         ConvertFromInternalShowFormats(vod.ShowFormats),
		ThumbnailTemplates:  ConvertInternalThumbnailTemplates(vod.ThumbnailTemplates),
		SeekPreviewsUrl:     vod.SeekPreviewsURL,
		AnimatedPreviewUrl:  vod.AnimatedPreviewURL,
		Path:                vod.Path,
		Url:                 vod.URL,
		HandwaveLatencyMs:   utils.ProtobufInt64Value(vod.HandwaveLatencyMS),
		Failure:             failure,
	}
}

// ConvertToInternalCreateArchiveRequest converts create archive request from rpc to internal
func ConvertToInternalCreateArchiveRequest(r *rpc.CreateArchiveRequest) *models.CreateArchiveRequest {
	var recordedOn *time.Time
	if r.RecordedOn != nil && r.RecordedOn.Seconds > 0 {
		recordedOn = utils.FromProtobufTimestampToTime(r.RecordedOn)
	}
	return &models.CreateArchiveRequest{
		BroadcastID:         utils.StrToInt64(r.BroadcastId),
		BroadcasterSoftware: r.BroadcasterSoftware,
		RecordedOn:          recordedOn,
		Title:               r.Title,
		ChannelID:           utils.StrToInt64(r.ChannelId),
		Duration:            r.Duration,
		URI:                 r.Uri,
		Manifest:            r.Manifest,
		Deleted:             r.Deleted,
		HandwaveLatencyMS:   r.HandwaveLatencyMs,
	}
}

// ConvertInternalVods converts Vods from internal representation to rpc
func ConvertInternalVods(vodsResponse *models.VodsResponse) []*rpc.Vod {
	vods := make([]*rpc.Vod, len(vodsResponse.Vods))
	for i, vod := range vodsResponse.Vods {
		vods[i] = ConvertInternalVod(vod)
	}
	return vods
}

// ConvertPublicVodAggregations converts public vod aggregations from internal representation to rpc
func ConvertPublicVodAggregations(vodAggregations *models.PublicVodAggregationsByIDsResponse) *rpc.PublicVodAggregationsByIDsResponse {
	return &rpc.PublicVodAggregationsByIDsResponse{
		TotalCount:    vodAggregations.TotalCount,
		TotalDuration: vodAggregations.TotalDuration,
	}
}

// ConvertVodPopularity converts vod popularity from internal representation to rpc
func ConvertVodPopularity(vodPopularity *models.VodPopularityResponse) *rpc.VodPopularityResponse {
	return &rpc.VodPopularityResponse{
		Popular: vodPopularity.Popular,
	}
}

// ConvertCreateError converts create error from internal representation to rpc
func ConvertCreateError(input *models.CreateErrorResponse) *rpc.CreateErrorResponse {
	return &rpc.CreateErrorResponse{
		VodId:        utils.Int64ToStr(input.VodID),
		ErrorCode:    input.ErrorCode,
		ErrorMessage: input.ErrorMessage,
		CreatedAt:    utils.ProtobufTimeAsTimestamp(&input.CreatedAt),
	}
}
