package server

import (
	"context"
	"errors"
	"testing"

	"code.justin.tv/vod/vodapi/internal/models"
	"code.justin.tv/vod/vodapi/internal/testdata"
	rpc "code.justin.tv/vod/vodapi/rpc/vodapi"
	. "github.com/smartystreets/goconvey/convey"
	. "github.com/stretchr/testify/mock"
	"github.com/twitchtv/twirp"
)

func TestManagerInternalGetVodsByUser(t *testing.T) {
	Convey("when calling ManagerInternalGetVodsByUser", t, func() {
		server, mocks := initTestableServer()
		ctx := context.Background()

		request := &rpc.ManagerInternalGetVodsByUserRequest{
			ChannelId: "1234",
			Language:  "english",
			Status:    rpc.ConvertInternalVodStatus("status"),
			Sort:      rpc.ConvertInternalVodSort("sort"),
			Limit:     100,
			Offset:    0,
		}

		Convey("when ManagerInternalGetVodsByUser succeeds", func() {
			rpcUpload := testdata.RPCUploadVod()
			uploadVod := testdata.UploadVod()
			vods := []*models.Vod{uploadVod}
			vodsResponse := &models.VodsResponse{
				Vods: vods,
			}
			mocks.backend.On("ManagerInternalGetVodsByUser", ctx, Anything).Return(vodsResponse, nil)
			resp, err := server.ManagerInternalGetVodsByUser(ctx, request)
			So(err, ShouldBeNil)
			So(resp, ShouldNotBeNil)
			So(len(resp.Vods), ShouldEqual, 1)
			So(resp.Vods[0], ShouldResemble, rpcUpload)
		})
		Convey("when ManagerInternalGetVodsByUser with broadcast type succeeds", func() {
			requestWithBroadcastType := &rpc.ManagerInternalGetVodsByUserRequest{
				ChannelId:      "1234",
				BroadcastTypes: []rpc.VodType{rpc.VodType_ARCHIVE},
				Language:       "english",
				Status:         rpc.ConvertInternalVodStatus("status"),
				Sort:           rpc.ConvertInternalVodSort("sort"),
				Limit:          100,
				Offset:         0,
			}

			rpcUpload := testdata.RPCUploadVod()
			uploadVod := testdata.UploadVod()
			vods := []*models.Vod{uploadVod}
			vodsResponse := &models.VodsResponse{
				Vods: vods,
			}
			mocks.backend.On("ManagerInternalGetVodsByUser", ctx, Anything).Return(vodsResponse, nil)
			resp, err := server.ManagerInternalGetVodsByUser(ctx, requestWithBroadcastType)
			So(err, ShouldBeNil)
			So(resp, ShouldNotBeNil)
			So(len(resp.Vods), ShouldEqual, 1)
			So(resp.Vods[0], ShouldResemble, rpcUpload)
		})
		Convey("when ManagerInternalGetVodsByUser fails", func() {
			mocks.MockTwirpErrorConvert()
			mocks.backend.On("ManagerInternalGetVodsByUser", ctx, Anything).Return(nil, errors.New("foo"))
			resp, err := server.ManagerInternalGetVodsByUser(ctx, request)
			So(resp, ShouldBeNil)
			So(err, ShouldBeError)
			twirpError := err.(twirp.Error)
			So(twirpError.Code(), ShouldEqual, twirp.Internal)
		})
		mocks.AssertExpectations(t)
	})
}
