package server

import (
	"log"
	"net/http"

	"github.com/cactus/go-statsd-client/statsd"
	"github.com/twitchtv/twirp"
	statsd_hook "github.com/twitchtv/twirp/hooks/statsd"
	goji "goji.io"
	"goji.io/pat"
	"golang.org/x/net/context"

	"code.justin.tv/foundation/twitchserver"
	"code.justin.tv/vod/vodapi/internal/auth"
	"code.justin.tv/vod/vodapi/internal/backend"
	"code.justin.tv/vod/vodapi/internal/errors"
	"code.justin.tv/vod/vodapi/internal/server/converters"
	"code.justin.tv/vod/vodapi/pkg/errorlogger"
	"code.justin.tv/vod/vodapi/rpc/vodapi"
)

var (
	// BuildTime is set to the current date when the app is built.
	BuildTime = "not provided"
	// Version is set to the current git SHA when the app is built.
	Version = "not provided"
)

const (
	defaultEndpointSampleRate = 1.0
)

// Server exposes HTTP endpoints.
type Server struct {
	*goji.Mux
	auth        auth.Handler
	backend     backend.Backend
	errorlogger errorlogger.ErrorLogger
	stats       statsd.Statter
}

// NewServer allocates and returns a new Server.
func NewServer(auth auth.Handler, backend backend.Backend, errorlogger errorlogger.ErrorLogger, stats statsd.Statter) *Server {
	server := twitchserver.NewServer()

	s := &Server{
		server,
		auth,
		backend,
		errorlogger,
		stats,
	}

	// Statsd hooks
	var basicHook *twirp.ServerHooks
	if stats != nil {
		basicHook = statsd_hook.NewStatsdServerHooks(stats)
	}
	basicHook.Error = func(ctx context.Context, err twirp.Error) context.Context {
		if err == nil {
			return ctx
		}

		code := err.Code()
		if code == twirp.Internal || code == twirp.Unimplemented || code == twirp.Unavailable || code == converters.InvalidTwirpErrorCode {
			log.Printf("Internal twirp error: %s\n", err.Error())
		}
		return ctx
	}
	hook := twirp.ChainHooks(basicHook)

	// Health check
	twitchserver.RegisterHealthCheckHook(s.health)

	// Auth handled by middleware.
	server.Use(auth.AuthMiddleware)

	// Register Twirp RPC endpoints.
	twirpHandler := vodapi.NewVodApiServer(s, hook)
	s.Handle(pat.Post(vodapi.VodApiPathPrefix+"*"), twirpHandler)

	return s
}

func (s *Server) getStatusCode(err error) int {
	// vodapi wrapped errors.
	if err, ok := err.(errors.WithStatusCode); ok {
		return err.StatusCode()
	}

	return http.StatusInternalServerError
}
