package server

import (
	e "errors"
	"net/http"
	"testing"

	"github.com/cactus/go-statsd-client/statsd"
	. "github.com/smartystreets/goconvey/convey"
	. "github.com/stretchr/testify/mock"

	authMock "code.justin.tv/vod/vodapi/internal/auth/mocks"
	backendMock "code.justin.tv/vod/vodapi/internal/backend/mocks"
	"code.justin.tv/vod/vodapi/internal/errors"
	errorloggerMock "code.justin.tv/vod/vodapi/pkg/errorlogger/mocks"
)

type serverMockClients interface {
	AssertExpectations(t *testing.T)
}

type serverMockClientsImpl struct {
	auth        *authMock.Handler
	backend     *backendMock.Backend
	errorlogger *errorloggerMock.ErrorLogger
}

// func (f *fakeHandler) AuthMiddleware(h http.Handler) http.Handler {
// 	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
// 		h.ServeHTTP(w, r)
// 	})
// }

func initTestableServer() (*Server, *serverMockClientsImpl) {
	auth := &authMock.Handler{}
	backend := &backendMock.Backend{}
	errorlogger := &errorloggerMock.ErrorLogger{}
	stats, _ := statsd.NewNoopClient()

	// var healthHandler = (*Server).health
	// auth.On("AuthMiddleware", Anything).Return(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) { h.ServeHTTP(w, r) }))

	auth.On("AuthMiddleware", Anything).Return(nil)
	server := NewServer(auth, backend, errorlogger, stats)

	mocks := &serverMockClientsImpl{
		auth:        auth,
		backend:     backend,
		errorlogger: errorlogger,
	}
	return server, mocks
}

func TestGetStatusCode(t *testing.T) {
	Convey("getStatusCode", t, func() {
		s, _ := initTestableServer()

		Convey("handles vodapi error", func() {
			err := &errors.GenericInternal{}
			So(s.getStatusCode(err), ShouldEqual, http.StatusInternalServerError)
		})

		Convey("handles generic error", func() {
			err := e.New("random 500")
			So(s.getStatusCode(err), ShouldEqual, http.StatusInternalServerError)
		})
	})
}

func (s *serverMockClientsImpl) MockTwirpErrorConvert() {
	s.errorlogger.On("LogErrorWithExtras", Anything, Anything, Anything)
}

func (s *serverMockClientsImpl) AssertExpectations(t *testing.T) {
	s.auth.AssertExpectations(t)
	s.backend.AssertExpectations(t)
	s.errorlogger.AssertExpectations(t)
}
