package testdata

import (
	"code.justin.tv/vod/vodapi/internal/server/converters"
	rpc "code.justin.tv/vod/vodapi/rpc/vodapi"
	"code.justin.tv/vod/vodapi/rpc/vodapi/utils"
)

// RPCUploadVod is the upload vod
func RPCUploadVod() *rpc.Vod {
	return converters.ConvertInternalVod(UploadVod())
}

// RPCArchiveVod is the archive vod
func RPCArchiveVod() *rpc.Vod {
	return converters.ConvertInternalVod(ArchiveVod())
}

// CreateUploadRequest returns a default upload request
func CreateUploadRequest() *rpc.CreateUploadRequest {
	vod := RPCUploadVod()
	return CreateUploadRequestFromVod(vod)
}

// CreateUploadRequestFromVod is the test create upload request
func CreateUploadRequestFromVod(vod *rpc.Vod) *rpc.CreateUploadRequest {
	return &rpc.CreateUploadRequest{
		BroadcastId:   vod.BroadcastId,
		BroadcastType: vod.BroadcastType,
		OwnerId:       vod.OwnerId,
		Description:   utils.ConvertStrPtrToStr(utils.FromProtobufStringValue(vod.Description)),
		Duration:      vod.Duration,
		Uri:           vod.Uri,
		Manifest:      utils.ConvertStrPtrToStr(utils.FromProtobufStringValue(vod.Manifest)),
		Title:         utils.ConvertStrPtrToStr(utils.FromProtobufStringValue(vod.Title)),
		Game:          utils.ConvertStrPtrToStr(utils.FromProtobufStringValue(vod.Game)),
		Communities:   vod.Communities,
		Language:      utils.ConvertStrPtrToStr(utils.FromProtobufStringValue(vod.Language)),
		TagList:       vod.TagList,
		Viewable:      vod.Viewable,
		ViewableAt:    utils.ProtobufTimeAsTimestamp(utils.TimePtr(utils.Now())),
	}
}

// CreateArchiveRequest is the test create archive request
func CreateArchiveRequest() *rpc.CreateArchiveRequest {
	vod := RPCArchiveVod()
	showFormats := make(map[string]*rpc.ShowFormatRequest, len(vod.ShowFormats))
	for key, format := range vod.ShowFormats {
		showFormats[key] = &rpc.ShowFormatRequest{
			PlaylistPreference: format.PlaylistPreference,
			DisplayName:        format.DisplayName,
			Bitrate:            format.Bitrate,
			Fps:                format.Fps,
			MaxIdrInterval:     format.MaxIdrInterval,
			Codecs:             format.Codecs,
			Resolution:         format.Resolution,
		}
	}
	var manifest string
	if vod.Manifest != nil {
		manifest = vod.Manifest.Value
	}

	return &rpc.CreateArchiveRequest{
		BroadcastId:         vod.BroadcastId,
		BroadcasterSoftware: "web",
		RecordedOn:          vod.StartedOn,
		ChannelId:           vod.OwnerId,
		Duration:            vod.Duration,
		Uri:                 vod.Url,
		Manifest:            manifest,
		Deleted:             false,
		HandwaveLatencyMs:   500,
	}
}

// CreateHighlightRequest is the test create highlight request
func CreateHighlightRequest() *rpc.ManagerCreateHighlightRequest {
	vod := RPCUploadVod()
	return &rpc.ManagerCreateHighlightRequest{
		VodId:        vod.Id,
		Description:  vod.Description.Value,
		Game:         vod.Game.Value,
		Title:        vod.Title.Value,
		Language:     vod.Language.Value,
		StartSeconds: 10,
		EndSeconds:   100,
		TagList:      vod.TagList,
		CreatedBy:    vod.OwnerId,
	}
}

// CreateThumbnailsRequest is the test create thumbnails request
func CreateThumbnailsRequest() *rpc.CreateThumbnailsRequest {
	return &rpc.CreateThumbnailsRequest{
		Thumbnails: []*rpc.ThumbnailRequest{
			&rpc.ThumbnailRequest{
				Path: "path",
			},
		},
	}
}

// CreateVodAppealsRequest is the test create appeals request
func CreateVodAppealsRequest() *rpc.ManagerCreateVodAppealsRequest {
	return &rpc.ManagerCreateVodAppealsRequest{
		VodId:          utils.Int64ToStr(VodID1),
		City:           "city",
		Country:        "country",
		FullName:       "full_name",
		State:          "state",
		StreetAddress1: "street address 1",
		StreetAddress2: "street address 2",
		Zipcode:        "zipcode",
		TrackAppeals: []*rpc.TrackAppealRequest{
			&rpc.TrackAppealRequest{
				AudibleMagicResponseId: "6666",
				Reason:                 "reason",
			},
		},
	}
}

// CreateAudibleMagicResponsesRequest is the test create audible magic response request
func CreateAudibleMagicResponsesRequest() *rpc.CreateAudibleMagicResponsesRequest {
	return &rpc.CreateAudibleMagicResponsesRequest{
		AudibleMagicResponses: []*rpc.CreateAudibleMagicResponseRequest{
			&rpc.CreateAudibleMagicResponseRequest{
				VodId:                utils.Int64ToStr(VodID1),
				Title:                "title",
				Performer:            "performer",
				Genre:                "genre",
				Artist:               "artist",
				AlbumTitle:           "album_title",
				Song:                 "song",
				Isrc:                 "isrc",
				IsMatch:              true,
				MatchOffsetSeconds:   10,
				MatchDurationSeconds: 20,
				ScanOffsetSeconds:    30,
				ScanDurationSeconds:  40,
				MuteOffsetSeconds:    50,
				MuteDurationSeconds:  60,
				AudibleMagicItemId:   "audible_magic_item_id",
			},
		},
	}
}
