package tracker

import (
	"log"

	"github.com/afex/hystrix-go/hystrix"
	"golang.org/x/net/context"

	"code.justin.tv/vod/vodapi/pkg/shared"
	"code.justin.tv/vod/vodapi/pkg/spade"
)

const (
	eventSourceBackend = "backend"
)

// Tracker exposes methods to track record operations
type Tracker interface {
}

type trackerImpl struct {
	spade spade.Client
}

type recordCreatedProperties struct {
	UserID    string `json:"user_id"`
	ChannelID string `json:"channel_id"`
	ID        string `json:"id"`
}

type recordUpdatedProperties struct {
	UserID    string `json:"user_id"`
	ChannelID string `json:"channel_id"`
	ID        string `json:"id"`
}

type recordDeletedProperties struct {
	UserID    string `json:"user_id"`
	ChannelID string `json:"channel_id"`
	ID        string `json:"id"`
}

// NewTracker initializes a new tracker
func NewTracker(spadeClient spade.Client) (Tracker, error) {
	return &trackerImpl{
		spade: spadeClient,
	}, nil
}

func (t *trackerImpl) track(ctx context.Context, eventName string, properties interface{}) {
	err := hystrix.Do("spade_track", func() error {
		return t.spade.Track(ctx, eventName, properties, nil)
	}, nil)
	if err != nil {
		log.Printf("error while sending %s event: %s, %+v", eventName, err.Error(), properties)
	}
}

func (t *trackerImpl) extractUserID(ctx context.Context) string {
	userID, ok := shared.UserIDFromContext(ctx)
	if !ok {
		log.Printf("error while extracting UserID for spade event: not found in context")
		return ""
	}

	return userID
}

// NoopTracker for staging/dev.
type NoopTracker struct{}
