package hystrixhelper

import (
	"log"
	"net/http"
	"time"

	"github.com/afex/hystrix-go/hystrix"
	"github.com/pkg/errors"
)

const (
	maxRetryCount   = 5
	maxRetryBackoff = 2000
)

var errHystrixInternal = errors.New("hystrix internal error")

// HystrixRoundTripWrapper wraps a roundtripper with hystrix.
type HystrixRoundTripWrapper struct {
	Next         http.RoundTripper
	Name         string
	RetryCount   int
	RetryBackoff int
}

// NewHystrixRoundTripWrapper returns a new HystrixRoundTripWrapper.
func NewHystrixRoundTripWrapper(name string, maxConcurrentRequests int, timeout int, retryCount int, retryBackoff int) func(c http.RoundTripper) http.RoundTripper {
	return func(next http.RoundTripper) http.RoundTripper {
		config := hystrix.CommandConfig{
			MaxConcurrentRequests: maxConcurrentRequests,
			Timeout:               timeout,
		}
		hystrix.ConfigureCommand(name, config)

		if retryCount > maxRetryCount {
			retryCount = maxRetryCount
		}
		if retryBackoff > maxRetryBackoff {
			retryBackoff = maxRetryBackoff
		}
		return &HystrixRoundTripWrapper{
			Next:         next,
			Name:         name,
			RetryCount:   retryCount,
			RetryBackoff: retryBackoff,
		}
	}
}

// RoundTrip wraps twitchclient round trip calls with hystrix.
func (h *HystrixRoundTripWrapper) RoundTrip(req *http.Request) (*http.Response, error) {
	var resp *http.Response
	err := hystrix.Do(h.Name, func() error {
		retryResp, retryError := h.roundTripWithRetries(req)
		resp = retryResp
		return retryError
	}, nil)

	if err != nil {
		// No error with the request just the 5xx error for circuit break count
		// Since this is a valid response, return as normal and let the caller deal with the status code parsing
		if err == errHystrixInternal {
			log.Printf("Hystrix encountered an HTTP 5XX from service: %s", h.Name)
			return resp, nil
		}
		// hystrix itself ran into issues so propogate this up
		return nil, errors.Wrap(err, "Error from hystrix circuit")
	}

	return resp, nil
}

func (h *HystrixRoundTripWrapper) roundTripWithRetries(req *http.Request) (*http.Response, error) {
	var lastResp *http.Response
	var lastError error
	// We will make h.RetryCount + 1 attempts to execute the roundtrip and only break if we get a non-5xx response
	// If we fail to make the request we will return the last error or 5xx response from the downstream service
	for i := 0; i <= h.RetryCount; i++ {
		// reset response and error before each try
		lastResp = nil
		lastError = nil

		resp, err := h.Next.RoundTrip(req)
		// the http protocol itself got an error
		if err != nil {
			lastError = errors.Wrap(err, "Error from round tripper")
		} else {
			lastResp = resp
			// 5xx should go towards the circuit break count
			if lastResp.StatusCode >= http.StatusInternalServerError {
				lastError = errHystrixInternal
			} else {
				// no error from protocol and wasn't a 500 so we don't need to try again, just stop retrying
				return lastResp, nil
			}
		}

		// wait some backoff time so we don't just hit the service one after the other
		time.Sleep(time.Duration(h.RetryBackoff) * time.Millisecond)
	}
	return lastResp, lastError
}
