package vinyl

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	"code.justin.tv/foundation/twitchclient"
	vinylclient "code.justin.tv/vod/vinyl/client"
)

const (
	defaultStatSampleRate = 0.1
)

// Client is a Vinyl client.
type Client interface {
	vinylclient.Client
	CreateUpload(ctx context.Context, vod CreateUploadVod, reqOpts *twitchclient.ReqOpts) (*vinylclient.Vod, error)
	CreateArchive(ctx context.Context, vod CreateArchiveVod, reqOpts *twitchclient.ReqOpts) (*vinylclient.Vod, error)
	InternalCreateHighlight(ctx context.Context, vod InternalCreateHighlightVod, reqOpts *twitchclient.ReqOpts) (*vinylclient.Vod, error)
	GetVodsByUserIncludeBannedUsers(ctx context.Context, input *GetVodsByUserIncludeBannedUsersInput, reqOpts *twitchclient.ReqOpts) ([]*vinylclient.Vod, error)
	GetVodsByIDsIncludeBannedUsers(ctx context.Context, ids []string, reqOpts *twitchclient.ReqOpts) ([]*vinylclient.Vod, error)
	UndeleteVods(ctx context.Context, ids []string, reqOpts *twitchclient.ReqOpts) ([]*vinylclient.Vod, error)
	SoftDeleteVodsInInterval(ctx context.Context, startTimeUnixSeconds string, endTimeUnixSeconds string, reqOpts *twitchclient.ReqOpts) error
	HardDeleteVods(ctx context.Context, ids []string, reqOpts *twitchclient.ReqOpts) ([]*vinylclient.Vod, error)
	FinalizeUpload(ctx context.Context, id string, reqOpts *twitchclient.ReqOpts) error
	SetViewcounts(ctx context.Context, viewcounts *SetViewcountsInput, reqOpts *twitchclient.ReqOpts) error
	UpdateUserVodProperties(ctx context.Context, updateUserVodProperties *UpdateUserVodPropertiesInput, reqOpts *twitchclient.ReqOpts) (*vinylclient.UserVODProperties, error)
	GetAppeals(ctx context.Context, getAppeals *GetAppealsInput, reqOpts *twitchclient.ReqOpts) (*GetAppealsResponse, error)
	ResolveTrackAppeal(ctx context.Context, trackAppealID string, action string, reqOpts *twitchclient.ReqOpts) error
	ResolveVodAppeal(ctx context.Context, vodAppealID string, reqOpts *twitchclient.ReqOpts) error
	UpdateAMRs(ctx context.Context, updateAudibleMagicResponses *UpdateAudibleMagicResponsesInput, reqOpts *twitchclient.ReqOpts) ([]*vinylclient.AMR, error)
}

type vinylWrapper struct {
	conf twitchclient.ClientConf
	vinylclient.Client
}

// NewClient returns a Vinyl client.
func NewClient(conf twitchclient.ClientConf) (Client, error) {
	client, err := vinylclient.NewClient(conf)
	if err != nil {
		return nil, err
	}
	return &vinylWrapper{
		conf,
		client,
	}, nil
}

// CreateUpload creates an upload vod
func (v *vinylWrapper) CreateUpload(ctx context.Context, vod CreateUploadVod, reqOpts *twitchclient.ReqOpts) (*vinylclient.Vod, error) {
	twitchClient, err := twitchclient.NewClient(v.conf)
	if err != nil {
		return nil, err
	}
	vodInput, err := json.Marshal(createUploadWrapper{vod})
	if err != nil {
		return nil, err
	}

	req, err := twitchClient.NewRequest(http.MethodPost, "/v1/vods", bytes.NewBuffer(vodInput))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.vinyl.create_upload",
		StatSampleRate: defaultStatSampleRate,
	})

	var responseVod createVodResponse
	_, err = twitchClient.DoJSON(ctx, &responseVod, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}
	return &responseVod.Vod, nil
}

// CreateArchive creates an archive vod
func (v *vinylWrapper) CreateArchive(ctx context.Context, vod CreateArchiveVod, reqOpts *twitchclient.ReqOpts) (*vinylclient.Vod, error) {
	twitchClient, err := twitchclient.NewClient(v.conf)
	if err != nil {
		return nil, err
	}
	vodInput, err := json.Marshal(createArchiveWrapper{vod})
	if err != nil {
		return nil, err
	}

	req, err := twitchClient.NewRequest(http.MethodPost, "/v1/vods", bytes.NewBuffer(vodInput))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.vinyl.create_archive",
		StatSampleRate: defaultStatSampleRate,
	})

	var responseVod createVodResponse
	_, err = twitchClient.DoJSON(ctx, &responseVod, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}
	return &responseVod.Vod, nil
}

// InternalCreateHighlight creates a highlight vod
func (v *vinylWrapper) InternalCreateHighlight(ctx context.Context, vod InternalCreateHighlightVod, reqOpts *twitchclient.ReqOpts) (*vinylclient.Vod, error) {
	twitchClient, err := twitchclient.NewClient(v.conf)
	if err != nil {
		return nil, err
	}
	vodInput, err := json.Marshal(internalCreateHighlightWrapper{vod})
	if err != nil {
		return nil, err
	}

	req, err := twitchClient.NewRequest(http.MethodPost, "/v1/vods", bytes.NewBuffer(vodInput))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.vinyl.internal_create_highlight",
		StatSampleRate: defaultStatSampleRate,
	})

	var responseVod createVodResponse
	_, err = twitchClient.DoJSON(ctx, &responseVod, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}
	return &responseVod.Vod, nil
}

// GetVodsByUserIncludeBannedUsers gets the vods by user include banned users
func (v *vinylWrapper) GetVodsByUserIncludeBannedUsers(ctx context.Context, input *GetVodsByUserIncludeBannedUsersInput, reqOpts *twitchclient.ReqOpts) ([]*vinylclient.Vod, error) {
	twitchClient, err := twitchclient.NewClient(v.conf)
	if err != nil {
		return nil, err
	}
	broadcastType := ""
	if input.BroadcastType != "" {
		broadcastType = fmt.Sprintf("&broadcast_type=%s", input.BroadcastType)
	}

	requestStr := fmt.Sprintf("/v1/vods/user/%s?appeals_and_amrs=true&include_unviewable=true&include_dmca_and_tos=true%s&offset=%d&limit=%d", input.ChannelID, broadcastType, input.Offset, input.Limit)
	req, err := twitchClient.NewRequest(http.MethodGet, requestStr, nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.vinyl.get_vods_by_user_include_banned_users",
		StatSampleRate: defaultStatSampleRate,
	})

	data := vodsResponse{}
	_, err = twitchClient.DoJSON(ctx, &data, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if len(data.Vods) == 0 {
		return nil, &twitchclient.Error{StatusCode: http.StatusNotFound, Message: "vods not found"}
	}

	return data.Vods, nil
}

// GetVodsByIDsIncludeBannedUsers gets the vods by ids include banned users
func (v *vinylWrapper) GetVodsByIDsIncludeBannedUsers(ctx context.Context, ids []string, reqOpts *twitchclient.ReqOpts) ([]*vinylclient.Vod, error) {
	twitchClient, err := twitchclient.NewClient(v.conf)
	if err != nil {
		return nil, err
	}
	idsStr := strings.Join(ids, ",")

	req, err := twitchClient.NewRequest(http.MethodGet, fmt.Sprintf("/v1/vods?ids=%s&appeals_and_amrs=true&include_unviewable=true&include_dmca_and_tos=true", idsStr), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.vinyl.get_vods_by_ids_include_banned_users",
		StatSampleRate: defaultStatSampleRate,
	})

	data := vodsResponse{}
	_, err = twitchClient.DoJSON(ctx, &data, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if len(data.Vods) == 0 {
		return nil, &twitchclient.Error{StatusCode: http.StatusNotFound, Message: "vods not found"}
	}

	return data.Vods, nil
}

// UndeleteVods undeletes the vods
func (v *vinylWrapper) UndeleteVods(ctx context.Context, ids []string, reqOpts *twitchclient.ReqOpts) ([]*vinylclient.Vod, error) {
	// NOTE: this should only be called by admin panel and for one vod at a time, so this usage pattern is fine
	//       since this logic is temporary, leaving as serial requests for now
	twitchClient, err := twitchclient.NewClient(v.conf)
	if err != nil {
		return nil, err
	}

	vinylVods := make([]*vinylclient.Vod, len(ids))
	for i, id := range ids {
		req, err := twitchClient.NewRequest(http.MethodPut, fmt.Sprintf("/v1/vods?ids=%s", id), bytes.NewBufferString("{\"deleted\": false}"))
		if err != nil {
			return nil, err
		}

		combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
			StatName:       "service.vinyl.undelete_vod",
			StatSampleRate: defaultStatSampleRate,
		})

		var responseVod createVodResponse
		_, err = twitchClient.DoJSON(ctx, &responseVod, req, combinedReqOpts)
		if err != nil {
			return nil, err
		}
		vinylVods[i] = &responseVod.Vod
	}

	return vinylVods, nil
}

// SoftDeleteVodsInInterval deletes the vods in interval
func (v *vinylWrapper) SoftDeleteVodsInInterval(ctx context.Context, startTimeUnixSeconds string, endTimeUnixSeconds string, reqOpts *twitchclient.ReqOpts) error {
	twitchClient, err := twitchclient.NewClient(v.conf)
	if err != nil {
		return err
	}

	req, err := twitchClient.NewRequest(http.MethodGet, fmt.Sprintf("/v1/vods?start_time=%s&end_time=%s", startTimeUnixSeconds, endTimeUnixSeconds), nil)
	if err != nil {
		return err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.vinyl.soft_delete_vods_in_interval",
		StatSampleRate: defaultStatSampleRate,
	})

	_, err = twitchClient.DoNoContent(ctx, req, combinedReqOpts)
	return err
}

// HardDeleteVods hard deletes the vods
func (v *vinylWrapper) HardDeleteVods(ctx context.Context, ids []string, reqOpts *twitchclient.ReqOpts) ([]*vinylclient.Vod, error) {
	twitchClient, err := twitchclient.NewClient(v.conf)
	if err != nil {
		return nil, err
	}
	idsStr := strings.Join(ids, ",")

	req, err := twitchClient.NewRequest(http.MethodDelete, fmt.Sprintf("/v1/vods_from_s3?ids=%s", idsStr), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.vinyl.hard_delete_vods",
		StatSampleRate: defaultStatSampleRate,
	})

	data := vodsResponse{}
	_, err = twitchClient.DoJSON(ctx, &data, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if len(data.Vods) == 0 {
		return nil, &twitchclient.Error{StatusCode: http.StatusNotFound, Message: "vods not found"}
	}

	return data.Vods, nil
}

// FinalizeUpload finalizes the upload
func (v *vinylWrapper) FinalizeUpload(ctx context.Context, id string, reqOpts *twitchclient.ReqOpts) error {
	twitchClient, err := twitchclient.NewClient(v.conf)
	if err != nil {
		return err
	}

	req, err := twitchClient.NewRequest(http.MethodPost, fmt.Sprintf("/v1/vods/%s/finalize_upload", id), nil)
	if err != nil {
		return err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.vinyl.finalize_upload",
		StatSampleRate: defaultStatSampleRate,
	})

	_, err = twitchClient.DoNoContent(ctx, req, combinedReqOpts)
	return err
}

// SetViewcounts sets the view counts
func (v *vinylWrapper) SetViewcounts(ctx context.Context, viewcounts *SetViewcountsInput, reqOpts *twitchclient.ReqOpts) error {
	// NOTE: this should only be called by countess and for one vod at a time, so this usage pattern is fine
	//       since this logic is temporary, leaving as serial requests for now
	twitchClient, err := twitchclient.NewClient(v.conf)
	if err != nil {
		return err
	}

	for _, viewcount := range viewcounts.Viewcounts {
		viewcountInput, err := json.Marshal(viewcount)
		if err != nil {
			return err
		}

		req, err := twitchClient.NewRequest(http.MethodPost, "/v1/vods/set_viewcounts", bytes.NewBuffer(viewcountInput))
		if err != nil {
			return err
		}

		combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
			StatName:       "service.vinyl.set_viewcounts",
			StatSampleRate: defaultStatSampleRate,
		})

		_, err = twitchClient.DoNoContent(ctx, req, combinedReqOpts)
		if err != nil {
			return err
		}
	}
	return nil
}

// UpdateUserVodProperties updates the user vod properties
func (v *vinylWrapper) UpdateUserVodProperties(ctx context.Context, updateUserVodProperties *UpdateUserVodPropertiesInput, reqOpts *twitchclient.ReqOpts) (*vinylclient.UserVODProperties, error) {
	twitchClient, err := twitchclient.NewClient(v.conf)
	if err != nil {
		return nil, err
	}

	updateUserVodPropertiesInput, err := json.Marshal(updateUserVodProperties)
	if err != nil {
		return nil, err
	}

	req, err := twitchClient.NewRequest(http.MethodPost, fmt.Sprintf("/v1/user_vod_properties/%s", updateUserVodProperties.ChannelID), bytes.NewBuffer(updateUserVodPropertiesInput))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.vinyl.update_user_vod_properties",
		StatSampleRate: defaultStatSampleRate,
	})

	var userVodProperties *vinylclient.UserVODProperties
	_, err = twitchClient.DoJSON(ctx, &userVodProperties, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}
	return userVodProperties, nil
}

// GetAppeals gets the appeals
func (v *vinylWrapper) GetAppeals(ctx context.Context, getAppeals *GetAppealsInput, reqOpts *twitchclient.ReqOpts) (*GetAppealsResponse, error) {
	twitchClient, err := twitchclient.NewClient(v.conf)
	if err != nil {
		return nil, err
	}

	query := url.Values{}
	if getAppeals.VodID != "" {
		query.Add("vod_id", getAppeals.VodID)
	}
	if getAppeals.UserInfo != "" {
		query.Add("user_info", getAppeals.UserInfo)
	}
	if getAppeals.Priority != nil {
		query.Add("priority", strconv.FormatBool(*getAppeals.Priority))
	}
	if getAppeals.Resolved != nil {
		query.Add("resolved", strconv.FormatBool(*getAppeals.Resolved))
	}
	if getAppeals.Limit != 0 {
		query.Add("limit", strconv.FormatInt(getAppeals.Limit, 10))
	}
	if getAppeals.Offset != 0 {
		query.Add("offset", strconv.FormatInt(getAppeals.Offset, 10))
	}

	req, err := twitchClient.NewRequest(http.MethodGet, fmt.Sprintf("/v1/vod_appeals?%s", query.Encode()), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.vinyl.get_appeals",
		StatSampleRate: defaultStatSampleRate,
	})

	data := GetAppealsResponse{}
	_, err = twitchClient.DoJSON(ctx, &data, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	return &data, nil
}

// ResolveTrackAppeal resolves the track appeal
func (v *vinylWrapper) ResolveTrackAppeal(ctx context.Context, trackAppealID string, action string, reqOpts *twitchclient.ReqOpts) error {
	twitchClient, err := twitchclient.NewClient(v.conf)
	if err != nil {
		return err
	}

	trackAppealBody := map[string]string{"action": action}
	trackAppealInput, err := json.Marshal(trackAppealBody)
	if err != nil {
		return err
	}

	req, err := twitchClient.NewRequest(http.MethodPost, fmt.Sprintf("/v1/track_appeal/%s", trackAppealID), bytes.NewBuffer(trackAppealInput))
	if err != nil {
		return err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.vinyl.resolve_track_appeal",
		StatSampleRate: defaultStatSampleRate,
	})

	_, err = twitchClient.DoNoContent(ctx, req, combinedReqOpts)
	return err
}

// ResolveVodAppeal resolves the vod appeal
func (v *vinylWrapper) ResolveVodAppeal(ctx context.Context, vodAppealID string, reqOpts *twitchclient.ReqOpts) error {
	twitchClient, err := twitchclient.NewClient(v.conf)
	if err != nil {
		return err
	}

	req, err := twitchClient.NewRequest(http.MethodPost, fmt.Sprintf("/v1/vod_appeal/%s", vodAppealID), nil)
	if err != nil {
		return err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.vinyl.resolve_vod_appeal",
		StatSampleRate: defaultStatSampleRate,
	})

	_, err = twitchClient.DoNoContent(ctx, req, combinedReqOpts)
	return err
}

// UpdateAMRs updates the audible magic responses
func (v *vinylWrapper) UpdateAMRs(ctx context.Context, updateAudibleMagicResponses *UpdateAudibleMagicResponsesInput, reqOpts *twitchclient.ReqOpts) ([]*vinylclient.AMR, error) {
	// NOTE: this should only be called by vod workers and for one vod at a time, so this usage pattern is fine
	//       since this logic is temporary, leaving as serial requests for now
	twitchClient, err := twitchclient.NewClient(v.conf)
	if err != nil {
		return nil, err
	}

	amrs := make([]*vinylclient.AMR, len(updateAudibleMagicResponses.UpdateAudibleMagicResponses))
	for i, updateAudibleMagicResponse := range updateAudibleMagicResponses.UpdateAudibleMagicResponses {
		updateAudibleMagicResponseInput, err := json.Marshal(updateAudibleMagicResponse)
		if err != nil {
			return nil, err
		}

		req, err := twitchClient.NewRequest(http.MethodPut, fmt.Sprintf("/v1/amrs/%d", updateAudibleMagicResponse.AudibleMagicResponseID), bytes.NewBuffer(updateAudibleMagicResponseInput))
		if err != nil {
			return nil, err
		}

		combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
			StatName:       "service.vinyl.update_audible_magic_response",
			StatSampleRate: defaultStatSampleRate,
		})

		data := &vinylclient.AMR{}
		_, err = twitchClient.DoJSON(ctx, &data, req, combinedReqOpts)
		if err != nil {
			return nil, err
		}

		amrs[i] = data
	}
	return amrs, nil
}
