package utils

import (
	"testing"
	"time"

	"code.justin.tv/vod/vodapi/rpc/vodapi/utils"
	"github.com/golang/protobuf/ptypes/timestamp"

	"github.com/golang/protobuf/ptypes/wrappers"
	"github.com/stretchr/testify/assert"
)

func TestProtobufBoolValue(t *testing.T) {
	val := true
	assert.Equal(t, &wrappers.BoolValue{Value: true}, ProtobufBoolValue(&val))
	assert.Nil(t, ProtobufBoolValue(nil))
}

func TestProtobufStringValue(t *testing.T) {
	val := "123"
	assert.Equal(t, &wrappers.StringValue{Value: "123"}, ProtobufStringValue(&val))
	assert.Nil(t, ProtobufStringValue(nil))
}

func TestProtobufInt64Value(t *testing.T) {
	val := int64(123)
	assert.Equal(t, &wrappers.Int64Value{Value: 123}, ProtobufInt64Value(&val))
	assert.Nil(t, ProtobufInt64Value(nil))
}

func TestProtobufFloat64Value(t *testing.T) {
	val := float64(123.12)
	assert.Equal(t, &wrappers.DoubleValue{Value: 123.12}, ProtobufFloat64Value(&val))
	assert.Nil(t, ProtobufFloat64Value(nil))
}

func TestProtobufTimeAsTimestamp(t *testing.T) {
	val := time.Date(2017, time.January, 1, 0, 0, 0, 0, time.UTC)
	assert.Equal(t, &timestamp.Timestamp{Seconds: 1483228800}, ProtobufTimeAsTimestamp(&val))
	assert.Nil(t, ProtobufTimeAsTimestamp(nil))
	assert.Nil(t, ProtobufTimeAsTimestamp(utils.TimePtr(time.Time{})))
}

func TestFromProtobufTimestampToTime(t *testing.T) {
	val := timestamp.Timestamp{Seconds: 1483228800}
	tm := time.Date(2017, time.January, 1, 0, 0, 0, 0, time.UTC)
	defaultTime := time.Unix(0, 0).UTC()
	assert.Equal(t, &tm, FromProtobufTimestampToTime(&val))
	assert.Equal(t, &defaultTime, FromProtobufTimestampToTime(nil))
}

func TestFromProtobufStringValue(t *testing.T) {
	val := wrappers.StringValue{Value: "abc"}
	str := "abc"
	assert.Equal(t, &str, FromProtobufStringValue(&val))
	assert.Nil(t, FromProtobufStringValue(nil))
}

func TestFromProtobufFloat64Value(t *testing.T) {
	val := wrappers.DoubleValue{Value: 123.12}
	num := 123.12
	assert.Equal(t, &num, FromProtobufFloat64Value(&val))
	assert.Nil(t, FromProtobufFloat64Value(nil))
}

func TestFromProtobufBoolValue(t *testing.T) {
	val := wrappers.BoolValue{Value: true}
	boolVal := true
	assert.Equal(t, &boolVal, FromProtobufBoolValue(&val))
	assert.Nil(t, FromProtobufBoolValue(nil))
}
