package utils

import (
	"testing"
	"time"

	"github.com/golang/protobuf/ptypes/timestamp"

	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/assert"
)

func TestNow(t *testing.T) {
	Convey("StubTimeNow", t, func() {
		Now()
		StubTimeNow()
		So(Now(), ShouldResemble, time.Date(2017, 12, 12, 12, 12, 0, 0, time.UTC))
	})
}

func TestGetTimeFromTimestamp(t *testing.T) {
	Convey("with nil timestamp", t, func() {
		assert.Equal(t, time.Unix(0, 0).UTC(), GetTimeFromTimestamp(nil))
	})
	Convey("with valid timestamp", t, func() {
		assert.Equal(t, time.Date(2017, time.January, 1, 0, 0, 0, 0, time.UTC), GetTimeFromTimestamp(&timestamp.Timestamp{Seconds: 1483228800}))
	})
}

func TestGetTimestampFromTimePtr(t *testing.T) {
	timeVal := time.Date(2017, time.January, 1, 0, 0, 0, 0, time.UTC)
	assert.Nil(t, GetTimestampFromTimePtr(nil))
	assert.Equal(t, &timestamp.Timestamp{Seconds: 1483228800}, GetTimestampFromTimePtr(&timeVal))
}

func TestGetTimestampFromTime(t *testing.T) {
	assert.Equal(t, &timestamp.Timestamp{Seconds: 1483228800}, GetTimestampFromTime(time.Date(2017, time.January, 1, 0, 0, 0, 0, time.UTC)))
}

func TestStrPtr(t *testing.T) {
	str := "abc"
	assert.Equal(t, &str, StrPtr(str))
}

func TestInt64Ptr(t *testing.T) {
	val := int64(123)
	assert.Equal(t, &val, Int64Ptr(val))
}

func TestTimePtr(t *testing.T) {
	val := time.Now()
	assert.Equal(t, &val, TimePtr(val))
}

func TestTimestampPtr(t *testing.T) {
	val := timestamp.Timestamp{Seconds: 100}
	assert.Equal(t, &val, TimestampPtr(val))
}

func TestConvertStrPtrToStr(t *testing.T) {
	str := "abc"
	assert.Equal(t, "abc", ConvertStrPtrToStr(&str))
	assert.Equal(t, "", ConvertStrPtrToStr(nil))
}

func TestStrToInt64(t *testing.T) {
	assert.Equal(t, int64(1234), StrToInt64("1234"))
	assert.Equal(t, int64(0), StrToInt64("abcd"))
}

func TestInt64ToStr(t *testing.T) {
	assert.Equal(t, "1234", Int64ToStr(int64(1234)))
}

func TestStrPtrToInt64Ptr(t *testing.T) {
	input := "1234"
	expected := int64(1234)
	assert.Equal(t, &expected, StrPtrToInt64Ptr(&input))
	assert.Nil(t, StrPtrToInt64Ptr(nil))
}

func TestInt64PtrToStrPtr(t *testing.T) {
	input := int64(1234)
	expected := "1234"
	assert.Equal(t, &expected, Int64PtrToStrPtr(&input))
	assert.Nil(t, Int64PtrToStrPtr(nil))
}

func TestStrArrayToInt64Array(t *testing.T) {
	assert.Equal(t, []int64{int64(1234)}, StrArrayToInt64Array([]string{"1234"}))
	assert.Equal(t, []int64{int64(2345), int64(3456), int64(4567)}, StrArrayToInt64Array([]string{"2345", "3456", "4567"}))
}
