package external_integrations

import (
	"errors"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
)

const (
	production = "production"
)

// Settings is the list of settings specific to the environment.
type Settings struct {
	OAuthToken    string
	TestUserID    string
	VisageHost    string
	DynamoDbTable string
	Data          *Data
}

// Data is the list of test data by environment.
type Data struct {
	Record1       TestRecord
	Record2       TestRecord
	RecordUpdate  TestRecordUpdate
	RecordsResult TestRecordsResult
}

// getSettings gets the environment specific settings.
func getSettings(environment string) *Settings {
	// As of May 1st 2017, oauth tokens don't expire, so instead of doing messy programmatic
	// 3rd party token granting, we're just going to hardcode the oauth token until the
	// process is simpler.
	if environment == production {
		return &Settings{
			OAuthToken:    "auhme51wku18j4due6tu39typxp3vc",
			TestUserID:    "157237187",
			VisageHost:    "https://visage.production.us-west2.justin.tv/",
			DynamoDbTable: "vodapi-production",
			Data: &Data{
				Record1:       testRecord1Production,
				Record2:       testRecord2Production,
				RecordUpdate:  testRecordUpdateProduction,
				RecordsResult: testRecordsResultProduction,
			},
		}
	}
	return &Settings{
		OAuthToken:    "j731u5osxj6dfbes2jsj17482t7yn8",
		TestUserID:    "155393848",
		VisageHost:    "https://visage.staging.us-west2.justin.tv/",
		DynamoDbTable: "vodapi-staging",
		Data: &Data{
			Record1:       testRecord1,
			Record2:       testRecord2,
			RecordUpdate:  testRecordUpdate,
			RecordsResult: testRecordsResult,
		},
	}
}

func recordEquals(first TestRecord, second TestRecord) bool {
	return first.RecordID == second.RecordID &&
		first.ChannelID == second.ChannelID &&
		first.Data == second.Data
}

func recordsResultsEquals(first TestRecordsResult, second TestRecordsResult) bool {
	if len(first.Records) != len(second.Records) {
		return false
	}

	for i, r := range first.Records {
		if !recordEquals(*r, *second.Records[i]) {
			return false
		}
	}

	return true
}

func teardown(settings *Settings) error {
	sess, err := session.NewSession(&aws.Config{
		Region: aws.String("us-west-2")},
	)
	if err != nil {
		return err
	}
	svc := dynamodb.New(sess)

	query := &dynamodb.QueryInput{
		TableName:              aws.String(settings.DynamoDbTable),
		IndexName:              aws.String("ChannelID"),
		KeyConditionExpression: aws.String("ChannelID = :ChannelID"),
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":ChannelID": {
				S: aws.String(settings.TestUserID),
			},
		},
	}
	result, err := svc.Query(query)
	if err != nil {
		return err
	}
	if len(result.Items) > 0 {
		var recordID string
		if result.Items[0]["ID"].S != nil {
			recordID = *result.Items[0]["ID"].S
		}
		if recordID == "" {
			return errors.New("Unable to parse record id")
		}

		_, err := svc.DeleteItem(&dynamodb.DeleteItemInput{
			TableName: aws.String(settings.DynamoDbTable),
			Key: map[string]*dynamodb.AttributeValue{
				"ID": {
					S: aws.String(recordID),
				},
			},
		})
		if err != nil {
			return err
		}
	}

	return nil
}
