package external_integrations

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/url"

	"code.justin.tv/foundation/twitchclient"
)

const (
	defaultStatSampleRate = 1.0
	defaultTimingXactName = "vodapi_test"
)

// Need a separate test client to hit visage endpoints
type vodapiClient struct {
	twitchclient.Client
	apiToken string
	userID   string
}

func newClient(host, apiToken, userID string) (*vodapiClient, error) {
	conf := twitchclient.ClientConf{
		Host: host,
	}

	if conf.TimingXactName == "" {
		conf.TimingXactName = defaultTimingXactName
	}

	twitchClient, err := twitchclient.NewClient(conf)
	if err != nil {
		return nil, err
	}

	return &vodapiClient{twitchClient, apiToken, userID}, nil
}

// POST on /channels/:channel_id/records
func (c vodapiClient) CreateRecord(testRecord TestRecord) (*TestRecord, error) {
	path := fmt.Sprintf("/channels/%s/records", url.QueryEscape(c.userID))

	input, err := json.Marshal(testRecord)
	if err != nil {
		return nil, err
	}

	req, err := c.NewRequest(http.MethodPost, path, bytes.NewBuffer(input))
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("OAuth %s", c.apiToken))
	req.Header.Add("Content-Type", "application/json")

	reqOpts := twitchclient.ReqOpts{
		StatName:       "service.vodapi.create_record",
		StatSampleRate: defaultStatSampleRate,
	}

	record := &TestRecord{}

	_, err = c.DoJSON(context.Background(), &record, req, reqOpts)
	if err != nil {
		return nil, err
	}

	return record, nil
}

// DELETE on /channels/:channel_id/records/:record_id
func (c vodapiClient) DeleteRecord(recordID string) error {
	path := fmt.Sprintf("/channels/%s/records/%s", url.QueryEscape(c.userID), url.QueryEscape(recordID))

	req, err := c.NewRequest(http.MethodDelete, path, nil)
	if err != nil {
		return err
	}
	req.Header.Add("Authorization", fmt.Sprintf("OAuth %s", c.apiToken))

	reqOpts := twitchclient.ReqOpts{
		StatName:       "service.vodapi.delete_record",
		StatSampleRate: defaultStatSampleRate,
	}

	_, err = c.doBetter(context.Background(), req, reqOpts)
	return err
}

// PUT on /channels/:channel_id/records/:record_id
func (c vodapiClient) UpdateRecord(testRecordUpdate TestRecordUpdate) error {
	path := fmt.Sprintf("/channels/%s/records/%s", url.QueryEscape(c.userID), url.QueryEscape(testRecordUpdate.RecordID))

	input, err := json.Marshal(testRecordUpdate)
	if err != nil {
		return err
	}

	req, err := c.NewRequest(http.MethodPut, path, bytes.NewBuffer(input))
	if err != nil {
		return err
	}
	req.Header.Add("Authorization", fmt.Sprintf("OAuth %s", c.apiToken))
	req.Header.Add("Content-Type", "application/json")

	reqOpts := twitchclient.ReqOpts{
		StatName:       "service.vodapi.update_record",
		StatSampleRate: defaultStatSampleRate,
	}

	_, err = c.doBetter(context.Background(), req, reqOpts)
	return err
}

// GET on /channels/:channel_id/records/:record_id
func (c vodapiClient) GetRecord(recordID string) (*TestRecord, error) {
	path := fmt.Sprintf("/channels/%s/records/%s", url.QueryEscape(c.userID), url.QueryEscape(recordID))

	req, err := c.NewRequest(http.MethodGet, path, nil)
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("OAuth %s", c.apiToken))
	req.Header.Add("Content-Type", "application/json")

	reqOpts := twitchclient.ReqOpts{
		StatName:       "service.vodapi.get_record",
		StatSampleRate: defaultStatSampleRate,
	}

	record := &TestRecord{}
	_, err = c.DoJSON(context.Background(), &record, req, reqOpts)
	if err != nil {
		return nil, err
	}

	return record, nil
}

// GET on /channels/:channel_id/records
func (c vodapiClient) GetRecords() (*TestRecordsResult, error) {
	path := fmt.Sprintf("/channels/%s/records", url.QueryEscape(c.userID))

	req, err := c.NewRequest(http.MethodGet, path, nil)
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("OAuth %s", c.apiToken))
	req.Header.Add("Content-Type", "application/json")

	reqOpts := twitchclient.ReqOpts{
		StatName:       "service.vodapi.get_records",
		StatSampleRate: defaultStatSampleRate,
	}

	recordsResult := &TestRecordsResult{}
	_, err = c.DoJSON(context.Background(), &recordsResult, req, reqOpts)
	if err != nil {
		return nil, err
	}

	return recordsResult, nil
}

func (c vodapiClient) doBetter(ctx context.Context, req *http.Request, opts twitchclient.ReqOpts) (*http.Response, error) {
	resp, err := c.Do(ctx, req, opts)
	if resp.StatusCode >= 500 {
		body, readAllErr := ioutil.ReadAll(resp.Body)
		if readAllErr != nil {
			return resp, errors.New(resp.Status)
		}
		return resp, errors.New(resp.Status + ": " + string(body))
	}

	if resp.StatusCode >= 400 {
		return resp, twitchclient.HandleFailedResponse(resp)
	}
	return resp, err
}
