package integrations

import (
	"context"
	"fmt"
	"os"
	"testing"

	"code.justin.tv/common/chitin"
	"code.justin.tv/vod/vodapi/internal/testdata"
	rpc "code.justin.tv/vod/vodapi/rpc/vodapi"
	"github.com/golang/protobuf/ptypes/wrappers"
	. "github.com/smartystreets/goconvey/convey"
)

func resetDynamicFields(originalVod *rpc.Vod, responseVod *rpc.Vod) {
	responseVod.CreatedAt = originalVod.CreatedAt
	responseVod.UpdatedAt = originalVod.UpdatedAt
}

func TestInternalRPCs(t *testing.T) {
	environment := os.Getenv("ENVIRONMENT")
	settings := getSettings(environment)
	host := os.Getenv("HOST")
	if host == "" {
		host = settings.Host
	}
	// TODO get token and put into context?
	ctx := context.Background()
	httpClient := chitin.Client(ctx)
	client := rpc.NewVodApiJSONClient(host, httpClient)
	upload := testdata.RPCUploadVod()
	upload.OwnerId = settings.TestUserID
	upload.CreatedBy = &wrappers.StringValue{Value: settings.TestUserID}
	// do not remove the Big Buck Bunny vod from vodapi_test user
	bigBuckBunny := &rpc.Vod{
		Id:    "290891473",
		Title: &wrappers.StringValue{Value: "BigBuckBunny.mp4"},
	}
	Convey("Internal Integration tests", t, func() {
		Convey("when cleaning up previous vods", func() {
			vods, err := client.ManagerGetVodsByUserIncludeBannedUsers(ctx, &rpc.ManagerGetVodsByUserIncludeBannedUsersRequest{
				ChannelId: settings.TestUserID,
				Limit:     10,
				Offset:    0,
			})
			So(err, ShouldBeNil)

			for _, vod := range vods.Vods {
				// don't delete the big buck bunny
				if vod.Id != bigBuckBunny.Id {
					_, err := client.InternalVodRemoveRecords(ctx, &rpc.InternalVodRemoveRecordsRequest{
						VodIds: []string{vod.Id},
					})
					So(err, ShouldBeNil)
				}
			}
		})

		Convey("CreateUpload", func() {
			uploadRequest := testdata.CreateUploadRequestFromVod(upload)
			uploadRequest.OwnerId = settings.TestUserID
			response, err := client.CreateUpload(ctx, uploadRequest)
			upload1AsRequest := testdata.CreateUploadRequestFromVod(response.Vod)
			upload1AsRequest.ViewableAt = uploadRequest.ViewableAt
			So(err, ShouldBeNil)
			So(upload1AsRequest, ShouldResemble, uploadRequest)
			So(response.Vod.Status, ShouldEqual, rpc.VodStatus_CREATED)
			upload = response.Vod
			fmt.Printf("Upload ID: %s\n\n", upload.Id)
		})

		Convey("GetPublicVodAggregationsByIDs", func() {
			vodAggregation, err := client.GetPublicVodAggregationsByIDs(ctx, &rpc.GetPublicVodAggregationsByIDsRequest{
				VodIds: []string{bigBuckBunny.Id},
			})
			So(err, ShouldBeNil)
			So(vodAggregation.GetTotalCount(), ShouldEqual, 1)
		})

		Convey("GetVodByIDIncludeBannedUsers", func() {
			// TODO: Ensure the user is banned for this functional test
			resp, err := client.GetVodByIDIncludeBannedUsers(ctx, &rpc.GetVodByIDIncludeBannedUsersRequest{
				VodId: bigBuckBunny.Id,
			})
			So(err, ShouldBeNil)
			So(resp.Vod.Title, ShouldResemble, bigBuckBunny.Title)
		})

		Convey("InternalGetVodByID", func() {
			// TODO: Ensure the video is hidden to functional test this function
			resp, err := client.InternalGetVodByID(ctx, &rpc.InternalGetVodByIDRequest{
				VodId: bigBuckBunny.Id,
			})
			So(err, ShouldBeNil)
			So(resp.Vod.Title, ShouldResemble, bigBuckBunny.Title)
		})

		Convey("PublicGetVodByID", func() {
			resp, err := client.PublicGetVodByID(ctx, &rpc.PublicGetVodByIDRequest{
				VodId: bigBuckBunny.Id,
			})
			resetDynamicFields(upload, resp.Vod)
			So(err, ShouldBeNil)
			So(resp.Vod.Title, ShouldResemble, bigBuckBunny.Title)
		})

		Convey("GetVodPopularity", func() {
			resp, err := client.GetVodPopularity(ctx, &rpc.GetVodPopularityRequest{
				VodId: bigBuckBunny.Id,
			})
			So(err, ShouldBeNil)
			So(resp.GetPopular(), ShouldBeFalse)
		})

		// TODO: enable after recording an archive
		// Convey("GetVodsByBroadcastIDsIncludeHidden", func() {
		// 	resp, err := client.GetVodsByBroadcastIDsIncludeHidden(ctx, &rpc.GetVodsByBroadcastIDsIncludeHiddenRequest{
		// 		BroadcastIds: []string{strconv.FormatInt(upload.BroadcastId, 10)},
		// 	})
		// 	So(err, ShouldBeNil)
		// 	So(resp, ShouldNotBeNil)
		// 	So(len(resp.Vods), ShouldEqual, 1)
		// })

		// TODO: Enable after adding big buck bunny to a community
		// Convey("GetVodsByCommunityIncludeHidden", func() {
		// 	resp, err := client.GetVodsByCommunityIncludeHidden(ctx, &rpc.GetVodsByCommunityIncludeHiddenRequest{
		// 		Community: testdata.HearthstoneCommunity,
		// 		Limit:     100,
		// 		Offset:    0,
		// 	})
		// 	So(err, ShouldBeNil)
		// 	So(resp, ShouldNotBeNil)
		// 	So(len(resp.Vods), ShouldEqual, 1)
		// })

		Convey("GetVodsByIDIncludeBannedUsers", func() {
			resp, err := client.GetVodByIDIncludeBannedUsers(ctx, &rpc.GetVodByIDIncludeBannedUsersRequest{
				VodId: bigBuckBunny.Id,
			})
			So(err, ShouldBeNil)
			So(resp.Vod.Title, ShouldResemble, bigBuckBunny.Title)
		})

		Convey("InternalGetVodsByIDs", func() {
			resp, err := client.InternalGetVodByID(ctx, &rpc.InternalGetVodByIDRequest{
				VodId: bigBuckBunny.Id,
			})
			So(err, ShouldBeNil)
			So(resp.Vod.Title, ShouldResemble, bigBuckBunny.Title)
		})

		Convey("PublicGetVodsByIDs", func() {
			resp, err := client.PublicGetVodsByIDs(ctx, &rpc.PublicGetVodsByIDsRequest{
				VodIds: []string{bigBuckBunny.Id},
			})
			So(err, ShouldBeNil)
			So(resp, ShouldNotBeNil)
			So(len(resp.Vods), ShouldEqual, 1)
			resetDynamicFields(upload, resp.Vods[0])
			So(resp.Vods[0].Title, ShouldResemble, bigBuckBunny.Title)
		})

		Convey("ManagerInternalGetVodsByUser", func() {
			resp, err := client.ManagerInternalGetVodsByUser(ctx, &rpc.ManagerInternalGetVodsByUserRequest{
				ChannelId: upload.OwnerId,
				Limit:     100,
				Offset:    0,
			})
			So(err, ShouldBeNil)
			So(resp, ShouldNotBeNil)
			bigBuckBunnyReteurned := false
			for _, vod := range resp.Vods {
				if vod.Id == bigBuckBunny.Id {
					So(vod.Title, ShouldResemble, bigBuckBunny.Title)
					bigBuckBunnyReteurned = true
				}
			}
			So(bigBuckBunnyReteurned, ShouldBeTrue)
		})

		Convey("PublicGetVodsByUser", func() {
			resp, err := client.PublicGetVodsByUser(ctx, &rpc.PublicGetVodsByUserRequest{
				ChannelId: upload.OwnerId,
				Limit:     100,
				Offset:    0,
			})
			So(err, ShouldBeNil)
			So(resp, ShouldNotBeNil)
			bigBuckBunnyReteurned := false
			for _, vod := range resp.Vods {
				if vod.Id == bigBuckBunny.Id {
					So(vod.Title, ShouldResemble, bigBuckBunny.Title)
					bigBuckBunnyReteurned = true
				}
			}
			So(bigBuckBunnyReteurned, ShouldBeTrue)
		})

		// 	Convey("SearchVods", func() {

		// 	})

		// 	Convey("SetViewcounts", func() {
		// 		resp, err := client.SetViewcounts(ctx, &rpc.SetViewcountsRequest{
		// 			Viewcounts: []*rpc.ViewcountsUpdate{
		// 				&rpc.ViewcountsUpdate{
		// 					VodId: upload.Id,
		// 					Count: 5,
		// 				},
		// 			},
		// 		})
		// 		So(err, ShouldBeNil)
		// 		So(resp, ShouldNotBeNil)

		// 		getVodResp, err := client.GetVodByID(ctx, &rpc.GetVodByIDRequest{
		// 			VodId: upload.Id,
		// 		})
		// 		So(err, ShouldBeNil)
		// 		So(getVodResp.Views, ShouldEqual, 5)
		// 	})

		// 	Convey("SoftDeleteVodsInInterval", func() {
		// 	})

		// depend on CreateUpload
		// TODO: remove dependency (maybe create a vod dedicated to being deleted / undeleted)
		Convey("SoftDeleteVods && UndeleteVods", func() {
			_, err := client.SoftDeleteVods(ctx, &rpc.SoftDeleteVodsRequest{
				VodIds: []string{upload.Id},
			})
			So(err, ShouldBeNil)

			// TODO: figure out why undelete vod doesn't work
			// resp, err := client.UndeleteVods(ctx, &rpc.UndeleteVodsRequest{
			// 	VodIds: []int64{upload.Id},
			// })
			// So(err, ShouldBeNil)
			// So(resp, ShouldNotBeNil)
			// So(len(resp.Vods), ShouldEqual, 1)
		})

		// 	Convey("UpdateAudibleMagicResponses", func() {

		// 	})

		// 	Convey("UpdateManifest", func() {

		// 	})

		Convey("ManagerUpdateVod", func() {
			resp, err := client.ManagerUpdateVod(ctx, &rpc.ManagerUpdateVodRequest{
				VodId: upload.Id,
				Title: &wrappers.StringValue{Value: "test title 2"},
			})
			So(err, ShouldBeNil)
			So(resp.Vod.Title.Value, ShouldEqual, "test title 2")
		})

		// 	Convey("YoutubeExport", func() {

		// 	})

		Convey("CreateThumbnails", func() {
			path := "path"
			thumbnailsResponse, err := client.CreateThumbnails(ctx, &rpc.CreateThumbnailsRequest{
				VodId: upload.Id,
				Thumbnails: []*rpc.ThumbnailRequest{
					&rpc.ThumbnailRequest{
						Path: path,
					},
				},
			})
			So(err, ShouldBeNil)

			found := false
			generatedThumbnailPaths := []string{}
			for _, thumbnail := range thumbnailsResponse.Thumbnails {
				if thumbnail.Path == path {
					found = true
					break
				} else {
					generatedThumbnailPaths = append(generatedThumbnailPaths, thumbnail.Path)
				}
			}
			So(found, ShouldBeTrue)

			Convey("DeleteThumbnails", func() {
				_, err = client.DeleteThumbnails(ctx, &rpc.DeleteThumbnailsRequest{
					VodId: upload.Id,
				})
				So(err, ShouldBeNil)

				receivedVod, err := client.PublicGetVodByID(ctx, &rpc.PublicGetVodByIDRequest{
					VodId: upload.Id,
				})
				So(err, ShouldBeNil)
				So(receivedVod, ShouldNotBeNil)

				foundMatch := false
				for _, thumbnail := range receivedVod.Vod.ThumbnailTemplates {
					for _, generatedThumbnailPath := range generatedThumbnailPaths {
						if generatedThumbnailPath == thumbnail.Path {
							foundMatch = true
							break
						}
					}
				}
				So(foundMatch, ShouldBeFalse)
			})
		})

		// keep this test last
		Convey("InternalVodRemoveRecords", func() {
			_, err := client.InternalVodRemoveRecords(ctx, &rpc.InternalVodRemoveRecordsRequest{
				VodIds: []string{upload.Id},
			})
			So(err, ShouldBeNil)
		})
	})
}
