package main

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"os"

	"github.com/aws/aws-lambda-go/lambda"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/codecommit"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/aws/aws-sdk-go/service/secretsmanager"
)

// SlackPayload defines slack's expected message
//
// Sending message
// For a simple message, your JSON payload could contain a text property at minimum. This is the text that will be posted to the channel.
//
// Adding links
// To create a link in your text, enclose the URL in <> angle brackets.
// For example: payload={"text": "<https://slack.com>"} will post a clickable link to https://slack.com.
//
// Customized Appearence
// You can customize the name and icon of your Incoming Webhook in the Integration Settings section below.
// However, you can override the displayed name by sending "username": "new-bot-name" in your JSON payload.
// You can also override the bot icon either with "icon_url": "https://slack.com/img/icons/app-57.png"
// or "icon_emoji": ":ghost:".
//
// Channel Override
// Incoming webhooks have a default channel, but it can be overridden in your JSON payload.
// A public channel can be specified with "channel": "#other-channel", and a Direct Message with "channel": "@username".

type CloudwatchEventInput struct {
	ID         string                      `json:"id"`
	DetailType string                      `json:"detail-type"`
	Source     string                      `json:"source"`
	Account    string                      `json:"account"`
	Time       string                      `json:"time"`
	Region     string                      `json:"region"`
	Detail     *CloudwatchEventInputDetail `json:"detail,omitempty"`
}

type CloudwatchEventInputDetail struct {
	Pipeline    string `json:"pipeline,omitempty"`
	State       string `json:"state,omitempty"`
	Stage       string `json:"stage,omitempty"`
	Action      string `json:"action,omitempty"`
	ExecutionID string `json:"execution-id,omitempty"`
}

type SlackPayload struct {
	Text        string             `json:"text,omitempty"`
	Channel     string             `json:"channel,omitempty"`
	Attachments []*SlackAttachment `json:"attachments,omitempty"`
	ThreadTs    string             `json:"thread_ts,omitempty"`
	AsUser      bool               `json:"as_user,omitempty"`
}

type SlackAttachment struct {
	Fallback   string `json:"fallback,omitempty"`
	Color      string `json:"color,omitempty"`
	AuthorName string `json:"author_name,omitempty"`
	AuthorLink string `json:"author_link,omitempty"`
	Title      string `json:"title,omitempty"`
	TitleLink  string `json:"title_link,omitempty"`
	Text       string `json:"text,omitempty"`
}

type SlackResponse struct {
	Ok *bool   `json:"ok"`
	Ts *string `json:"ts,omitempty"`
}

type DynamoRow struct {
	ExecutionID string `dynamodbav:"ExecutionID" json:"execution_id"`
	ThreadTs    string `dynamodbav:"ThreadTs" json:"thread_ts"`
}

const (
	repoNameKey        = "REPO_NAME"
	orgNameKey         = "ORG_NAME"
	deployTableNameKey = "DEPLOY_TABLE_NAME"
	slackChannelKey    = "SLACK_CHANNEL"
	slackURL           = "https://slack.com/api/chat.postMessage"
)

var (
	errInvalidStatusCode = errors.New("invalid status code")
	errEnvVarNotFound    = errors.New("missing env variables")
	errInvalidInput      = errors.New("input from cloudwatch event malformed")
	errSlackPost         = errors.New("error from slack post")
	errSlackPostTs       = errors.New("no timestamp from slack post")
	errDynamoRead        = errors.New("error reading from dynamo")

	colorMappings = map[string]string{
		"STARTED":    "good",
		"SUCCEEDED":  "good",
		"RESUMED":    "warning",
		"FAILED":     "danger",
		"CANCELED":   "danger",
		"SUPERSEDED": "warning",
	}

	cc  *codecommit.CodeCommit
	ddb *dynamodb.DynamoDB

	slackChannel    string
	repoName        string
	orgName         string
	deployTableName string
)

func handleRequest(ctx context.Context, inputStruct CloudwatchEventInput) error {

	sess, err := session.NewSession(&aws.Config{
		Region: aws.String("us-west-2"),
	})
	if err != nil {
		log.Fatal(err)
	}

	secrets := secretsmanager.New(sess)
	if err != nil {
		log.Fatal(err)
	}
	output, err := secrets.GetSecretValue(&secretsmanager.GetSecretValueInput{
		SecretId: aws.String("slack-notifier-token"),
	})
	if err != nil {
		log.Fatal(err)
	}
	slackToken := *output.SecretString

	detail := inputStruct.Detail
	var payload SlackPayload

	inputPayload, _ := json.Marshal(detail)
	log.Printf("input details:")
	log.Printf(fmt.Sprintf("%s", inputPayload))

	if detail.Stage == "" && detail.State == "STARTED" {
		commitHash, commitAuthor, err := getCodeCommitInfo(detail.Pipeline)
		payload = generateFirstMessage(detail, commitHash, commitAuthor)
		log.Printf("first message")
		resp, err := postToSlack(slackToken, payload)
		if err != nil {
			return err
		} else if resp.Ts == nil {
			return errSlackPostTs
		}
		writeDynamoRow(DynamoRow{ExecutionID: detail.ExecutionID, ThreadTs: *resp.Ts})
	} else {
		row, err := readDynamoRow(detail.ExecutionID)
		if err != nil {
			log.Printf(err.Error())
			return errDynamoRead
		}
		ts := row.ThreadTs
		if detail.Action == "" && detail.Stage == "" {
			payload = generatePipelineMessage(detail, ts)
			log.Printf("pipeline message")
		} else if detail.Action == "" {
			payload = generateStageMessage(detail, ts)
			log.Printf("stage message")
		} else {
			payload = generateActionMessage(detail, ts)
			log.Printf("action message")
		}
		_, err = postToSlack(slackToken, payload)
		if err != nil {
			log.Printf(err.Error())
			return err
		}
	}

	log.Printf("message sent")
	return nil
}

func postToSlack(slackToken string, payload SlackPayload) (*SlackResponse, error) {
	payloadJSON, _ := json.Marshal(payload)
	log.Printf("message to slack:")
	log.Printf(fmt.Sprintf("%s", payloadJSON))

	client := &http.Client{}
	r, _ := http.NewRequest("POST", slackURL, bytes.NewBuffer(payloadJSON))
	r.Header.Add("Authorization", fmt.Sprintf("Bearer %s", slackToken))
	r.Header.Add("Content-Type", "application/json")
	resp, err := client.Do(r)

	if err != nil {
		log.Printf("error making request to slack")
		return nil, err
	}
	if resp.StatusCode != http.StatusOK {
		log.Printf("post to slack had bad status code")
		return nil, errInvalidStatusCode
	}
	respStruct := SlackResponse{}
	respBodyBytes, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		log.Printf("can't read response body from slack")
		return nil, errSlackPost
	}
	log.Printf(fmt.Sprintf("%s", respBodyBytes))
	err = json.Unmarshal(respBodyBytes, &respStruct)
	if err != nil {
		log.Printf("can't unmarshal body from slack")
		return nil, err
	}
	return &respStruct, nil
}

func generateFirstMessage(detail *CloudwatchEventInputDetail, commitHash string, authorEmail string) SlackPayload {
	attachment := SlackAttachment{
		Fallback:   fmt.Sprintf("%s start", detail.Pipeline),
		AuthorName: fmt.Sprintf("CodePipeline: %s", detail.Pipeline),
		AuthorLink: fmt.Sprintf("https://us-west-2.console.aws.amazon.com/codepipeline/home?region=us-west-2#/view/%s", detail.Pipeline),
		Title:      fmt.Sprintf("Commit: %s", commitHash),
		TitleLink:  fmt.Sprintf("https://git-aws.internal.justin.tv/%s/%s/commit/%s", orgName, repoName, commitHash),
		Text:       fmt.Sprintf("Author: %s", authorEmail),
	}
	payload := SlackPayload{
		AsUser:      true,
		Attachments: []*SlackAttachment{&attachment},
		Channel:     slackChannel,
	}
	return payload
}

func generateStageMessage(detail *CloudwatchEventInputDetail, threadTs string) SlackPayload {
	attachment := SlackAttachment{
		Fallback:   fmt.Sprintf("%s: %s - %s", detail.Pipeline, detail.Stage, detail.State),
		Color:      colorMappings[detail.State],
		AuthorName: fmt.Sprintf("CodePipeline: %s", detail.Pipeline),
		AuthorLink: fmt.Sprintf("https://us-west-2.console.aws.amazon.com/codepipeline/home?region=us-west-2#/view/%s", detail.Pipeline),
		Title:      fmt.Sprintf("Stage %s: %s", detail.Stage, detail.State),
	}
	payload := SlackPayload{
		AsUser:      true,
		Attachments: []*SlackAttachment{&attachment},
		Channel:     slackChannel,
		ThreadTs:    threadTs,
	}
	return payload
}

func generateActionMessage(detail *CloudwatchEventInputDetail, threadTs string) SlackPayload {
	attachment := SlackAttachment{
		Fallback:   fmt.Sprintf("%s: %s, %s - %s", detail.Pipeline, detail.Stage, detail.Action, detail.State),
		Color:      colorMappings[detail.State],
		AuthorName: fmt.Sprintf("CodePipeline: %s", detail.Pipeline),
		AuthorLink: fmt.Sprintf("https://us-west-2.console.aws.amazon.com/codepipeline/home?region=us-west-2#/view/%s", detail.Pipeline),
		Title:      fmt.Sprintf("Stage %s, Action %s: %s", detail.Stage, detail.Action, detail.State),
	}
	payload := SlackPayload{
		AsUser:      true,
		Attachments: []*SlackAttachment{&attachment},
		Channel:     slackChannel,
		ThreadTs:    threadTs,
	}
	return payload
}

func generatePipelineMessage(detail *CloudwatchEventInputDetail, threadTs string) SlackPayload {
	attachment := SlackAttachment{
		Fallback:   fmt.Sprintf("%s: %s", detail.Pipeline, detail.State),
		Color:      colorMappings[detail.State],
		AuthorName: fmt.Sprintf("CodePipeline: %s", detail.Pipeline),
		AuthorLink: fmt.Sprintf("https://us-west-2.console.aws.amazon.com/codepipeline/home?region=us-west-2#/view/%s", detail.Pipeline),
		Title:      fmt.Sprintf("Pipeline: %s", detail.State),
	}
	payload := SlackPayload{
		AsUser:      true,
		Attachments: []*SlackAttachment{&attachment},
		Channel:     slackChannel,
		ThreadTs:    threadTs,
	}
	return payload
}

func getCodeCommitInfo(pipelineName string) (string, string, error) {
	branchName := "master"
	repoNamePtr := repoName
	branch := codecommit.GetBranchInput{BranchName: &branchName, RepositoryName: &repoNamePtr}
	branchOutput, err := cc.GetBranch(&branch)
	if err != nil || branchOutput.Branch == nil {
		return "", "", errors.New("Error reading from codecommit repo")
	}
	commitHash := branchOutput.Branch.CommitId
	commit := codecommit.GetCommitInput{CommitId: commitHash, RepositoryName: &repoNamePtr}
	commitOutput, err := cc.GetCommit(&commit)
	if err != nil || commitOutput.Commit == nil || commitOutput.Commit.Author == nil {
		return "", "", errors.New("Error reading from codecommit repo")
	}
	commitAuthor := commitOutput.Commit.Author
	if commitAuthor.Email == nil {
		return "", "", errors.New("Error reading from codecommit repo")
	}
	return *commitHash, *commitAuthor.Email, nil
}

func writeDynamoRow(row DynamoRow) error {
	recordMap, err := dynamodbattribute.MarshalMap(row)
	if err != nil {
		return fmt.Errorf("Error marshalling to dynamo")
	}

	input := &dynamodb.PutItemInput{
		TableName: aws.String(deployTableName),
		Item:      recordMap,
	}
	_, err = ddb.PutItem(input)
	return err
}

func readDynamoRow(executionID string) (*DynamoRow, error) {
	input := &dynamodb.GetItemInput{
		TableName: aws.String(deployTableName),
		Key:       map[string]*dynamodb.AttributeValue{"ExecutionID": {S: aws.String(executionID)}},
	}
	dynamoRecord, err := ddb.GetItem(input)
	if err != nil {
		return nil, err
	}

	if len(dynamoRecord.Item) == 0 {
		return nil, errors.New("No row found")
	}

	row := &DynamoRow{}
	err = dynamodbattribute.UnmarshalMap(dynamoRecord.Item, row)
	if err != nil {
		return nil, fmt.Errorf("Error unmarshalling from dynamo")
	}
	return row, nil
}

func main() {
	sess, err := session.NewSession(&aws.Config{
		Region: aws.String("us-west-2"),
	})
	if err != nil {
		panic(err)
	}
	cc = codecommit.New(sess)
	sess, err = session.NewSession(&aws.Config{
		Region: aws.String("us-west-2"),
	})
	if err != nil {
		panic(err)
	}
	ddb = dynamodb.New(sess)

	found := false
	slackChannel, found = os.LookupEnv(slackChannelKey)
	if !found {
		panic(errEnvVarNotFound)
	}
	repoName, found = os.LookupEnv(repoNameKey)
	if !found {
		panic(errEnvVarNotFound)
	}
	orgName, found = os.LookupEnv(orgNameKey)
	if !found {
		panic(errEnvVarNotFound)
	}
	deployTableName, found = os.LookupEnv(deployTableNameKey)
	if !found {
		panic(errEnvVarNotFound)
	}

	lambda.Start(handleRequest)
}
