// Helper library to wrap an http.Handler in statsd and chitin middleware.

package handler

import (
	"log"
	"net"
	"net/http"
	"os"

	"code.justin.tv/common/chitin"
	"code.justin.tv/common/golibs/statsd"
	"github.com/Hebo/loghandler"
	"github.com/zenazn/goji/graceful"
	"golang.org/x/net/context"
)

// New wraps an http.Handler with chitin and logging handlers.
func New(h http.Handler, stats statsd.Stats) (http.Handler, error) {
	// TODO: send stuff to trace once it's implemented on the chitin side.
	tracectx, err := chitin.ExperimentalTraceContext(context.Background())
	if err != nil {
		return nil, err
	}
	handler := chitin.Handler(h, chitin.SetBaseContext(tracectx))
	return loghandler.NewLoggingHandler(os.Stdout, stats, handler), nil
}

// BeginServer starts listening on an address with graceful restarts.
func BeginServer(h http.Handler, addr string) {
	graceful.HandleSignals()

	l, err := net.Listen("tcp", addr)
	if err != nil {
		log.Fatalf("[handler] Failed to bind to %s: %s", addr, err)
	}
	log.Printf("[handler] Listening on %s", addr)

	err = graceful.Serve(l, h)
	if err != nil {
		log.Fatalf("[handler] %s", err)
	}
	log.Printf("[handler] Draining server")
	graceful.Wait()
	log.Printf("[handler] Draining complete")
}
