package main

import (
	"flag"
	"log"

	"os"
	"strconv"

	_ "expvar"
	"net/http"
	_ "net/http/pprof"

	"code.justin.tv/common/golibs/statsd"
	"code.justin.tv/web/service_template/db"
	"github.com/getsentry/raven-go"

	"code.justin.tv/web/service_template/api"
	"code.justin.tv/web/service_template/backend"
	"code.justin.tv/web/service_template/counts"
	"code.justin.tv/web/service_template/handler"
)

var (
	addr        = ":8000"
	env         = "development"
	serviceName = "service_template"
)

func main() {
	flag.StringVar(&addr, "addr", addr, "Bind address")
	flag.StringVar(&env, "env", env, "Environment")

	flag.Parse()

	// Remove timestamps from log entries
	log.SetFlags(0)

	stats := configureStatsd()

	b := makeBackend(stats)
	apiMux := api.NewAPI(
		serviceName,
		b,
		stats,
		configureRaven(),
	)

	http.Handle("/", apiMux)

	h, err := handler.New(http.DefaultServeMux, stats)
	if err != nil {
		log.Fatalf("Failed to initialize handler: %s", err.Error())
	}
	handler.BeginServer(h, addr)
}

func configureStatsd() statsd.Stats {
	var stats statsd.Stats
	var err error

	statsdHostport := os.Getenv("STATSD_HOSTPORT")
	if len(statsdHostport) > 0 {
		if stats, err = statsd.Dial("udp", statsdHostport,
			statsd.StatsConfig{Rate: 0.1, Prefix: serviceName + "." + env}); err != nil {
			log.Fatalf("StatsD configuration error: %v", err)
		}
		log.Printf("Connected to StatsD at %s\n", statsdHostport)
	} else {
		stats = statsd.Noop()
	}
	return stats
}

func configureRaven() *raven.Client {
	var ravenClient *raven.Client
	var err error
	if dsn := os.Getenv("RAVEN_DSN"); len(dsn) > 0 {
		ravenClient, err = raven.NewClient(dsn, nil)
		if err != nil {
			log.Fatalf("Error creating raven client: %s\n", err.Error())
		} else {
			log.Printf("Created raven client\n")
		}
	} else {
		log.Println("Not logging to Sentry (Missing RAVEN_DSN var)")
	}
	return ravenClient
}

func makeBackend(stats statsd.Stats) *backend.Backend {
	masterPort, err := strconv.Atoi(os.Getenv("PG_MASTER_PORT"))
	if err != nil {
		log.Fatalf("PG_MASTER_PORT should be an integer")
	}

	master, err := db.OpenDB(
		os.Getenv("PG_DBNAME"),
		os.Getenv("PG_USER"),
		os.Getenv("PG_PASS"),
		os.Getenv("PG_MASTER_HOST"),
		masterPort)

	if err != nil {
		log.Fatalf("Failed to create db %s", err.Error())
	}

	b, err := backend.NewBackend(
		master,
		createCountsClient(stats),
		stats,
	)
	if err != nil {
		log.Fatalf("Error creating backend: %v\n", err)
	}

	return b
}

func createCountsClient(stats statsd.Stats) counts.Client {
	countessURL := os.Getenv("COUNTESS_BASE_URL")
	if len(countessURL) == 0 {
		log.Fatalln("Missing COUNTESS_BASE_URL")
	}
	return counts.NewClient(countessURL, stats)
}
