package api

import (
	"net/http"
	"net/http/httptest"
	"net/url"
	"strconv"
	"testing"

	"code.justin.tv/web/discovery/backend/backendtest"
	. "github.com/smartystreets/goconvey/convey"
)

func performAddAliasRequest(params url.Values) *httptest.ResponseRecorder {
	be := backendtest.NewBackend()
	router := API{Backend: be}

	r, _ := http.NewRequest("PUT", baseURL, nil)
	r.URL.RawQuery = params.Encode()

	return makeTestRequestWithContext(router.addAlias, r)
}

func TestAddAlias(t *testing.T) {
	Convey("should require alias", t, func() {
		params := url.Values{}
		w := performAddAliasRequest(params)

		So(w.Code, ShouldEqual, http.StatusBadRequest)
		So(w.Body.String(), ShouldContainSubstring, "Missing param: alias")
	})

	Convey("should require another param", t, func() {
		params := url.Values{
			"alias": []string{backendtest.TestAlias},
		}
		w := performAddAliasRequest(params)

		So(w.Code, ShouldEqual, http.StatusBadRequest)
		So(w.Body.String(), ShouldContainSubstring, "Requires 1 param")
	})

	Convey("should reject non int id arg", t, func() {
		params := url.Values{
			"alias": []string{backendtest.TestAlias},
			"id":    []string{backendtest.TestTerm},
		}
		w := performAddAliasRequest(params)

		So(w.Code, ShouldEqual, http.StatusBadRequest)
		So(w.Body.String(), ShouldContainSubstring, "Malformed id param")
	})

	Convey("should succeed for good id", t, func() {
		params := url.Values{
			"alias": []string{backendtest.TestAlias},
			"id":    []string{strconv.Itoa(backendtest.TestID)},
		}
		w := performAddAliasRequest(params)

		So(w.Code, ShouldEqual, http.StatusOK)
	})

	Convey("should return a game for good name", t, func() {
		params := url.Values{
			"alias": []string{backendtest.TestAlias},
			"name":  []string{backendtest.TestTerm},
		}
		w := performAddAliasRequest(params)

		So(w.Code, ShouldEqual, http.StatusOK)
	})

	Convey("should return not found for bad name", t, func() {
		params := url.Values{
			"alias": []string{backendtest.TestAlias},
			"name":  []string{backendtest.TestBadTerm},
		}
		w := performAddAliasRequest(params)

		So(w.Code, ShouldEqual, http.StatusNotFound)
	})
}
