package api

import (
	"fmt"
	"net/http"
	"strconv"

	"code.justin.tv/common/chitin"
	"code.justin.tv/web/discovery/backend"
	"code.justin.tv/web/discovery/game"
	"code.justin.tv/web/discovery/giantbomb"
	"github.com/jixwanwang/apiutils"
)

func (T *API) addGame(w http.ResponseWriter, r *http.Request) {
	ctx := chitin.Context(w, r)

	var err error
	var g giantbomb.GBGame
	if len(r.FormValue("giantbomb_id")) > 0 {
		id, err := strconv.Atoi(r.FormValue("giantbomb_id"))
		if err != nil {
			T.serveError(w, r, apiutils.NewErrorResponse(
				http.StatusBadRequest,
				fmt.Sprintf("Malformed giantbomb_id parameter: %s", r.FormValue("giantbomb_id")),
			))
			return
		}

		if T.Backend.IsGameBlacklisted(ctx, id) {
			T.serveError(w, r, apiutils.NewErrorResponse(http.StatusBadRequest, "Game is blacklisted from giantbomb"))
			return
		}

		g, err = T.Giantbomb.FetchGame(ctx, id)
		if err == giantbomb.ErrGameNotFound {
			T.serveError(w, r, apiutils.NewErrorResponse(http.StatusNotFound, err.Error()))
			return
		} else if err != nil {
			T.serveError(w, r, apiutils.NewErrorResponse(http.StatusInternalServerError, err.Error()))
			return
		}

		existing, err := T.Backend.GameByGBID(ctx, g.GiantbombID)
		// Try to find by name if there's no existing giantbomb id
		if err == backend.ErrGameNotFound {
			existing, err = T.Backend.GameByName(ctx, g.Name)
		}

		if err == nil {
			if r.FormValue("sync_name") == "true" {
				existing.Name = g.Name
			}

			existing.GiantbombID = g.GiantbombID
			existing.Images = g.Images
			err = T.Backend.UpdateGame(ctx, existing)
			if err != nil {
				T.serveError(w, r, apiutils.NewErrorResponse(http.StatusInternalServerError,
					fmt.Sprintf("Could not update game %s from giantbomb: %s", g.Name, err.Error())))
			} else {
				apiutils.ServeJSON(w, existing)
			}
			return
		}
	} else if len(r.FormValue("name")) > 0 {
		_, err = T.Backend.GameByName(ctx, r.FormValue("name"))
		if err == nil {
			T.serveError(w, r, apiutils.NewErrorResponse(422, "Game already exists"))
			return
		}

		g = giantbomb.GBGame{Game: game.NewGame()}
		g.Game.Name = r.FormValue("name")
	} else {
		T.serveError(w, r, apiutils.NewErrorResponse(http.StatusBadRequest, "Requires either giantbomb_id or name parameter"))
		return
	}

	err = T.Backend.AddGame(ctx, g.Game)
	if err != nil {
		T.serveError(w, r, apiutils.NewErrorResponse(http.StatusInternalServerError,
			fmt.Sprintf("Could not add game %s: %s", g.Name, err.Error())))
		return
	}

	apiutils.ServeJSON(w, g)
}
