package api

import (
	"encoding/json"
	"net/http"
	"net/http/httptest"
	"net/url"
	"strconv"
	"testing"

	"code.justin.tv/web/discovery/backend/backendtest"
	"code.justin.tv/web/discovery/game"
	"code.justin.tv/web/discovery/giantbomb/giantbombtest"
	. "github.com/smartystreets/goconvey/convey"
)

func performAddGameRequest(params url.Values) *httptest.ResponseRecorder {
	be := backendtest.NewBackend()
	router := API{Backend: be, Giantbomb: giantbombtest.NewGBClient()}

	r, _ := http.NewRequest("PUT", baseURL, nil)
	r.URL.RawQuery = params.Encode()

	return makeTestRequestWithContext(router.addGame, r)
}

func TestAddGame(t *testing.T) {
	Convey("should require either giantbomb id or name", t, func() {
		params := url.Values{}
		w := performAddGameRequest(params)

		So(w.Code, ShouldEqual, http.StatusBadRequest)
		So(w.Body.String(), ShouldContainSubstring, "Requires either giantbomb_id or name")
	})

	Convey("game should exist", t, func() {
		params := url.Values{
			"name": []string{backendtest.TestTerm},
		}
		w := performAddGameRequest(params)

		So(w.Code, ShouldEqual, 422)
	})

	Convey("should create empty game", t, func() {
		params := url.Values{
			"name": []string{backendtest.TestBadTerm},
		}
		w := performAddGameRequest(params)

		So(w.Code, ShouldEqual, http.StatusOK)
		So(w.Body.String(), ShouldContainSubstring, backendtest.TestBadTerm)
	})

	Convey("should reject non int giantbomb_id arg", t, func() {
		params := url.Values{
			"giantbomb_id": []string{backendtest.TestAlias},
		}
		w := performAddGameRequest(params)

		So(w.Code, ShouldEqual, http.StatusBadRequest)
		So(w.Body.String(), ShouldContainSubstring, "Malformed giantbomb_id param")
	})

	Convey("should not index blacklisted game", t, func() {
		params := url.Values{
			"giantbomb_id": []string{strconv.Itoa(backendtest.TestBlacklistedGBID)},
		}
		w := performAddGameRequest(params)

		So(w.Code, ShouldEqual, http.StatusBadRequest)
		So(w.Body.String(), ShouldContainSubstring, "blacklisted")
	})

	Convey("should update game, but not properties and name", t, func() {
		params := url.Values{
			"giantbomb_id": []string{strconv.Itoa(giantbombtest.TestGBGame.GiantbombID)},
		}
		w := performAddGameRequest(params)

		So(w.Code, ShouldEqual, http.StatusOK)

		var g game.Game

		dec := json.NewDecoder(w.Body)
		dec.Decode(&g)

		// should not update
		So(g.Name, ShouldEqual, backendtest.TestGame(giantbombtest.TestGBGame.ID).Name)
		So(g.Properties, ShouldResemble, backendtest.TestGame(giantbombtest.TestGBGame.ID).Properties)

		// should update
		So(g.GiantbombID, ShouldEqual, giantbombtest.TestGBGame.Game.GiantbombID)
		So(g.Images, ShouldResemble, giantbombtest.TestGBGame.Game.Images)
	})

	Convey("should update game name with sync_name = true", t, func() {
		params := url.Values{
			"giantbomb_id": []string{strconv.Itoa(giantbombtest.TestGBGame.GiantbombID)},
			"sync_name":    []string{"true"},
		}
		w := performAddGameRequest(params)

		So(w.Code, ShouldEqual, http.StatusOK)

		var g game.Game

		dec := json.NewDecoder(w.Body)
		dec.Decode(&g)

		// should not update
		So(g.Properties, ShouldResemble, backendtest.TestGame(giantbombtest.TestGBGame.ID).Properties)

		// should update
		So(g.GiantbombID, ShouldEqual, giantbombtest.TestGBGame.Game.GiantbombID)
		So(g.Images, ShouldResemble, giantbombtest.TestGBGame.Game.Images)
		So(g.Name, ShouldEqual, giantbombtest.TestGBGame.Game.Name)
	})
}
