package api

import (
	"encoding/json"
	"net/http"
	"net/http/httptest"
	"net/url"
	"strconv"
	"testing"

	"code.justin.tv/web/discovery/backend/backendtest"
	"code.justin.tv/web/discovery/game"
	. "github.com/smartystreets/goconvey/convey"
)

func performAddLocalizationRequest(params url.Values) *httptest.ResponseRecorder {
	be := backendtest.NewBackend()
	router := API{Backend: be}

	r, _ := http.NewRequest("PUT", baseURL, nil)
	r.URL.RawQuery = params.Encode()

	return makeTestRequestWithContext(router.addLocalization, r)
}

func TestAddLocalization(t *testing.T) {
	Convey("should require localized_name", t, func() {
		params := url.Values{
			"locale": []string{backendtest.TestLocale},
		}
		w := performAddLocalizationRequest(params)

		So(w.Code, ShouldEqual, http.StatusBadRequest)
		So(w.Body.String(), ShouldContainSubstring, "Missing param: localized_name")
	})

	Convey("should require locale", t, func() {
		params := url.Values{
			"localized_name": []string{backendtest.TestLocalizedName},
		}
		w := performAddLocalizationRequest(params)

		So(w.Code, ShouldEqual, http.StatusBadRequest)
		So(w.Body.String(), ShouldContainSubstring, "Missing param: locale")
	})

	Convey("should add localization from game_id", t, func() {
		params := url.Values{
			"localized_name": []string{backendtest.TestLocalizedName},
			"locale":         []string{backendtest.TestLocale},
			"game_id":        []string{strconv.Itoa(backendtest.TestID)},
		}
		w := performAddLocalizationRequest(params)

		So(w.Code, ShouldEqual, http.StatusOK)

		var l game.Localizations

		dec := json.NewDecoder(w.Body)
		dec.Decode(&l)

		So(l.GameID, ShouldEqual, backendtest.TestID)
		So(l.LocalizedNames, ShouldResemble, backendtest.TestLocalizedNames)
	})

	Convey("should add localization from game_name", t, func() {
		params := url.Values{
			"localized_name": []string{backendtest.TestLocalizedName},
			"locale":         []string{backendtest.TestLocale},
			"game_name":      []string{backendtest.TestTerm},
		}
		w := performAddLocalizationRequest(params)

		So(w.Code, ShouldEqual, http.StatusOK)

		var l game.Localizations

		dec := json.NewDecoder(w.Body)
		dec.Decode(&l)

		So(l.GameID, ShouldEqual, backendtest.TestID)
		So(l.LocalizedNames, ShouldResemble, backendtest.TestLocalizedNames)
	})
}
