package api

import (
	"fmt"
	"net/http"
	"strconv"

	"golang.org/x/net/context"

	"github.com/cactus/go-statsd-client/statsd"
	"github.com/getsentry/raven-go"
	"github.com/jixwanwang/apiutils"
	"github.com/zenazn/goji/web"

	"code.justin.tv/common/chitin"
	"code.justin.tv/common/golibs/errorlogger"
	"code.justin.tv/web/discovery/backend"
	"code.justin.tv/web/discovery/giantbomb"
)

// 15 minutes = 900 seconds
const defaultMaxCacheAge = 900

// API is an http.Handler that handles the http endpoints.
type API struct {
	Backend     backend.Backender
	Stats       statsd.Statter
	Raven       *raven.Client
	Giantbomb   giantbomb.Client
	ErrorLogger errorlogger.ErrorLogger
}

// NewAPI creates a routes.RouteMux and attaches the routes to it.
// The RouteMux can be attached to a handler (e.g. `http.Handle("/", apiMux)`)
func NewAPI(
	b *backend.Backend,
	stats statsd.Statter,
	raven *raven.Client,
	gb giantbomb.Client,
	logger errorlogger.ErrorLogger,
) (http.Handler, error) {
	router := API{
		Backend:     b,
		Stats:       stats,
		Raven:       raven,
		Giantbomb:   gb,
		ErrorLogger: logger,
	}
	mux := web.New()

	ctx, err := chitin.ExperimentalTraceContext(context.Background())
	if err != nil {
		return nil, err
	}

	mux.Use(func(h http.Handler) http.Handler {
		return chitin.Handler(
			h,
			chitin.SetBaseContext(ctx),
		)
	})

	// deprecated: use /game/suggest instead
	mux.Get("/suggest", router.createHandler(router.gameSuggest))

	mux.Get("/game", router.createHandler(router.game))
	mux.Get("/game/list", router.createHandler(router.gamesList))
	mux.Get("/game/suggest", router.createHandler(router.gameSuggest))
	mux.Put("/games", router.createHandler(router.addGame))
	mux.Put("/add_alias", router.createHandler(router.addAlias))
	mux.Get("/aliases", router.createHandler(router.getAllAliases))
	mux.Patch("/game/game/properties", router.createHandler(router.updateProperties))
	mux.Delete("/game", router.createHandler(router.deleteGame))

	mux.Get("/localizations", router.createHandler(router.getLocalizations))
	mux.Put("/localizations", router.createHandler(router.addLocalization))
	mux.Delete("/localizations", router.createHandler(router.deleteLocalization))
	mux.Delete("/localizations/all", router.createHandler(router.deleteLocalizations))

	mux.Get("/", router.createHandler(router.index))
	mux.Get("/debug/health", router.createHandler(router.health))
	return mux, nil
}

type handlerFunc func(http.ResponseWriter, *http.Request)

func (T *API) createHandler(handler handlerFunc) http.HandlerFunc {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		defer func() {
			if p := recover(); p != nil {
				if T.ErrorLogger != nil {
					T.ErrorLogger.RequestPanic(r, p)
				}
			}
		}()

		handler(w, r)
	})
}

// Health returns the state for use in debugging or health checks
func (T *API) health(w http.ResponseWriter, r *http.Request) {
	ctx := chitin.Context(w, r)

	err := T.Backend.Health(ctx)
	if err != nil {
		w.WriteHeader(http.StatusInternalServerError)
		fmt.Fprintf(w, "%s", err.Error())
		return
	}

	w.WriteHeader(http.StatusOK)
	fmt.Fprintf(w, "%s", "OK")
}

func shouldReport(errRes apiutils.ErrorResponse) bool {
	return errRes.Status >= 500
}

func (T *API) serveError(w http.ResponseWriter, r *http.Request, errRes apiutils.ErrorResponse) {
	apiutils.ServeError(w, errRes)

	if T.ErrorLogger != nil && shouldReport(errRes) {
		T.ErrorLogger.RequestError(r, errRes)
	}

	if T.Raven != nil && errRes.Status >= 500 {
		tags := map[string]string{
			"status": strconv.Itoa(errRes.Status),
		}
		T.Raven.CaptureError(errRes, tags, raven.NewHttp(r))
	}
}
