package api

import (
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/common/chitin"
	"code.justin.tv/web/discovery/backend/backendtest"
	"github.com/jixwanwang/apiutils"
)

const baseURL = "http://localhost/"

func TestWriteErrorToClient(t *testing.T) {
	w := httptest.NewRecorder()
	e := apiutils.NewErrorResponse(http.StatusBadRequest, "it broke")

	be := backendtest.NewBackend()
	router := API{Backend: be}

	router.serveError(w, &http.Request{}, e)

	if w.Code != e.Status {
		t.Errorf("wrong status code: have %d, want %d", w.Code, e.Status)
	}

	exp := `{
  "status": 400,
  "message": "it broke",
  "error": "Bad Request"
}`

	if w.Body.String() != exp {
		t.Errorf("wrong body: have %#v, want %#v", w.Body.String(), exp)
	}

	expType := "application/json"

	if w.Header().Get("content-type") != expType {
		t.Errorf("wrong content-type: have %#v, want %#v", w.Header().Get("content-type"), expType)
	}
}

func TestHealth(t *testing.T) {
	be := backendtest.NewBackend()
	router := API{Backend: be}

	r, _ := http.NewRequest("GET", baseURL, nil)
	w := makeTestRequestWithContext(router.health, r)

	if w.Code != http.StatusOK {
		t.Errorf("wrong status code: have %d, want %d", w.Code, http.StatusOK)
	}
}

func makeTestRequestWithContext(h func(http.ResponseWriter, *http.Request), r *http.Request) *httptest.ResponseRecorder {
	w := httptest.NewRecorder()
	// Add a chitin handler for context
	handler := chitin.Handler(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		h(w, r)
	}))
	handler.ServeHTTP(w, r)

	return w
}
