package api

import (
	"net/http"
	"net/http/httptest"
	"net/url"
	"strconv"
	"testing"

	"code.justin.tv/web/discovery/backend/backendtest"
	. "github.com/smartystreets/goconvey/convey"
)

func performDeleteGameRequest(params url.Values) *httptest.ResponseRecorder {
	be := backendtest.NewBackend()
	router := API{Backend: be}

	r, _ := http.NewRequest("DELETE", baseURL, nil)
	r.URL.RawQuery = params.Encode()

	return makeTestRequestWithContext(router.deleteGame, r)
}

func TestDeleteGame(t *testing.T) {
	Convey("should require one param", t, func() {
		params := url.Values{}
		w := performDeleteGameRequest(params)

		So(w.Code, ShouldEqual, http.StatusBadRequest)
		So(w.Body.String(), ShouldContainSubstring, "Requires 1 param")
	})

	Convey("should reject non int id arg", t, func() {
		params := url.Values{
			"id": []string{backendtest.TestBadTerm},
		}
		w := performDeleteGameRequest(params)

		So(w.Code, ShouldEqual, http.StatusBadRequest)
		So(w.Body.String(), ShouldContainSubstring, "Malformed id param")
	})

	Convey("should return a 204 for good id", t, func() {
		params := url.Values{
			"id": []string{strconv.Itoa(backendtest.TestID)},
		}
		w := performDeleteGameRequest(params)

		So(w.Code, ShouldEqual, http.StatusNoContent)
	})

	Convey("should return a 204 for good name", t, func() {
		params := url.Values{
			"name": []string{backendtest.TestTerm},
		}
		w := performDeleteGameRequest(params)

		So(w.Code, ShouldEqual, http.StatusNoContent)
	})

	Convey("should return a 204 for good alias", t, func() {
		params := url.Values{
			"name": []string{backendtest.TestAlias},
		}
		w := performDeleteGameRequest(params)

		So(w.Code, ShouldEqual, http.StatusNoContent)
	})

	Convey("should return a 204 for good giantbomb id", t, func() {
		params := url.Values{
			"giantbomb_id": []string{strconv.Itoa(backendtest.TestGBID)},
		}
		w := performDeleteGameRequest(params)

		So(w.Code, ShouldEqual, http.StatusNoContent)
	})

	Convey("should return not found for bad name", t, func() {
		params := url.Values{
			"name": []string{backendtest.TestBadTerm},
		}
		w := performDeleteGameRequest(params)

		So(w.Code, ShouldEqual, http.StatusNotFound)
	})
}
