package api

import (
	"golang.org/x/net/context"

	"code.justin.tv/web/discovery/game"
)

const (
	defaultLocale = "en-us"
)

// localizeGames takes in an array of game objects and add localization data to them.
// Result is either the same length as the input or an error.
// Does not attempt to localize for empty or default locale.
func (T *API) localizeGames(ctx context.Context, games []game.Game, locale string) ([]*Game, error) {
	if locale == "" || locale == defaultLocale {
		localized := []*Game{}
		for _, g := range games {
			localized = append(localized, &Game{
				Game:          g,
				LocalizedName: g.Name,
				Locale:        locale,
			})
		}
		return localized, nil
	}

	gameIDs := []int{}
	for _, g := range games {
		gameIDs = append(gameIDs, g.ID)
	}

	localizations, err := T.Backend.GetBulkLocalizations(ctx, gameIDs, locale)
	if err != nil {
		return nil, err
	}

	localized := []*Game{}
	defaultLocalization := game.Localizations{LocalizedNames: map[string]string{}}
	for _, g := range games {
		localization, ok := localizations[g.ID]
		if !ok {
			localization = defaultLocalization
		}

		localizedName, ok := localization.LocalizedNames[locale]
		if !ok {
			localizedName = g.Name
		}

		localized = append(localized, &Game{
			Game:          g,
			LocalizedName: localizedName,
			Locale:        locale,
		})
	}

	return localized, nil
}
