package api

import (
	"context"
	"fmt"
	"net/http"
	"net/url"
	"strconv"

	"code.justin.tv/web/discovery/backend"
	"github.com/jixwanwang/apiutils"
)

func (T *API) validateGameID(ctx context.Context, gameID int) *apiutils.ErrorResponse {
	_, err := T.Backend.GameByID(ctx, gameID)
	if err != nil {
		if err == backend.ErrGameNotFound {
			return &apiutils.ErrorResponse{
				Status:     http.StatusNotFound,
				Message:    fmt.Sprintf("Game ID not found: %v", gameID),
				StatusText: http.StatusText(http.StatusNotFound),
			}
		}
		return &apiutils.ErrorResponse{
			Status:     http.StatusInternalServerError,
			Message:    err.Error(),
			StatusText: http.StatusText(http.StatusInternalServerError),
		}
	}
	return nil
}

func (T *API) extractIDFromGameName(ctx context.Context, gameName string) (int, *apiutils.ErrorResponse) {
	var gameID int

	g, err := T.Backend.GameByNameOrAlias(ctx, gameName)
	if err != nil {
		if err == backend.ErrGameNotFound {
			return gameID, &apiutils.ErrorResponse{
				Status:     http.StatusNotFound,
				Message:    fmt.Sprintf("Game name not found: %v", gameName),
				StatusText: http.StatusText(http.StatusNotFound),
			}
		}
		return gameID, &apiutils.ErrorResponse{
			Status:     http.StatusInternalServerError,
			Message:    err.Error(),
			StatusText: http.StatusText(http.StatusInternalServerError),
		}
	}
	return g.ID, nil
}

func (T *API) getGameIDFromParams(ctx context.Context, params url.Values) (int, *apiutils.ErrorResponse) {
	var (
		gameID      int
		err         error
		errResponse *apiutils.ErrorResponse
	)

	if len(params["game_id"]) > 0 {
		gameID, err = strconv.Atoi(params.Get("game_id"))
		if err != nil {
			return gameID, &apiutils.ErrorResponse{
				Status:     http.StatusBadRequest,
				Message:    "Malformed game_id parameter",
				StatusText: http.StatusText(http.StatusBadRequest),
			}
		}
		errResponse = T.validateGameID(ctx, gameID)
		if errResponse != nil {
			return gameID, errResponse
		}

	} else if len(params["game_name"]) > 0 {
		gameName := params.Get("game_name")
		gameID, errResponse = T.extractIDFromGameName(ctx, gameName)
		if errResponse != nil {
			return gameID, errResponse
		}
	} else {
		return gameID, &apiutils.ErrorResponse{
			Status:     http.StatusBadRequest,
			Message:    "Either game_id or game_name must be provided",
			StatusText: http.StatusText(http.StatusBadRequest),
		}
	}
	return gameID, nil
}
