package api

import (
	"net/http"
	"net/http/httptest"
	"net/url"
	"strconv"
	"testing"

	"code.justin.tv/common/chitin"
	"code.justin.tv/web/discovery/backend/backendtest"
	. "github.com/smartystreets/goconvey/convey"
)

func getGameIDFromParams(params url.Values) (*httptest.ResponseRecorder, int, error) {
	be := backendtest.NewBackend()
	router := API{Backend: be}

	r, _ := http.NewRequest("GET", baseURL, nil)
	r.URL.RawQuery = params.Encode()
	w := httptest.NewRecorder()

	var (
		gameID int
		err    error
	)

	handler := chitin.Handler(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		gameID, err = router.getGameIDFromParams(chitin.Context(w, r), params)
	}))
	handler.ServeHTTP(w, r)

	return w, gameID, err
}

func GetGameIDFromParamsTest(t *testing.T) {
	Convey("should require game_name or game_id", t, func() {
		params := url.Values{}
		w, _, err := getGameIDFromParams(params)

		So(w.Code, ShouldEqual, http.StatusBadRequest)
		So(w.Body.String(), ShouldContainSubstring, "Either game_id or game_name must be provided")
		So(err, ShouldNotBeNil)
	})

	Convey("should reject non-int game_id argument", t, func() {
		params := url.Values{
			"game_id": []string{backendtest.TestBadTerm},
		}
		w, _, err := getGameIDFromParams(params)

		So(w.Code, ShouldEqual, http.StatusBadRequest)
		So(w.Body.String(), ShouldContainSubstring, "Malformed game_id parameter")
		So(err, ShouldNotBeNil)
	})

	Convey("should return not found for bad game_id", t, func() {
		params := url.Values{
			"game_id": []string{"322"},
		}
		w, _, err := getGameIDFromParams(params)

		So(w.Code, ShouldEqual, http.StatusNotFound)
		So(w.Body.String(), ShouldContainSubstring, "Game ID not found")
		So(err, ShouldNotBeNil)
	})

	Convey("should return not found for bad game_name", t, func() {
		params := url.Values{
			"game_name": []string{backendtest.TestBadTerm},
		}
		w, _, err := getGameIDFromParams(params)

		So(w.Code, ShouldEqual, http.StatusNotFound)
		So(w.Body.String(), ShouldContainSubstring, "Game name not found")
		So(err, ShouldNotBeNil)
	})

	Convey("should return success for valid game_id", t, func() {
		params := url.Values{
			"game_id": []string{strconv.Itoa(backendtest.TestID)},
		}
		w, gameID, err := getGameIDFromParams(params)

		So(w.Code, ShouldEqual, http.StatusOK)
		So(gameID, ShouldEqual, backendtest.TestID)
		So(err, ShouldBeNil)
	})

	Convey("should extract game ID for valid game_name", t, func() {
		params := url.Values{
			"game_name": []string{backendtest.TestTerm},
		}
		w, gameID, err := getGameIDFromParams(params)

		So(w.Code, ShouldEqual, http.StatusOK)
		So(gameID, ShouldEqual, backendtest.TestTerm)
		So(err, ShouldBeNil)
	})
}
