package api

import (
	"encoding/json"
	"net/http"
	"net/http/httptest"
	"net/url"
	"testing"

	"code.justin.tv/web/discovery/backend/backendtest"
	. "github.com/smartystreets/goconvey/convey"
)

func getSearchParams(term, live string) url.Values {
	return url.Values{
		"term": {term},
		"live": {live},
	}
}

func performRequest(params url.Values) *httptest.ResponseRecorder {
	be := backendtest.NewBackend()
	router := API{Backend: be}

	r, _ := http.NewRequest("GET", baseURL, nil)
	r.URL.RawQuery = params.Encode()

	return makeTestRequestWithContext(router.gameSuggest, r)
}

func TestSearch(t *testing.T) {
	Convey("should return empty json for missing params", t, func() {
		params := getSearchParams(backendtest.TestTerm, "")
		params.Del("term")
		w := performRequest(params)

		So(w.Code, ShouldEqual, http.StatusBadRequest)
		So(w.Body.String(), ShouldContainSubstring, "Missing param: term")
	})

	Convey("should return empty json for non-utf8 term", t, func() {
		params := getSearchParams(string([]byte{0xff}), "")
		w := performRequest(params)

		So(w.Code, ShouldEqual, http.StatusBadRequest)
		So(w.Body.String(), ShouldContainSubstring, "Non-UTF8 term")
	})

	Convey("should return empty json for no games", t, func() {
		params := getSearchParams(backendtest.TestBadTerm, "")
		w := performRequest(params)

		So(w.Code, ShouldEqual, http.StatusOK)
		So(w.Body.String(), ShouldEqual, "[]")
	})

	Convey("should correct json for a game", t, func() {
		params := getSearchParams(backendtest.TestTerm, "")
		w := performRequest(params)

		So(w.Code, ShouldEqual, http.StatusOK)
		var g []*Game

		dec := json.NewDecoder(w.Body)
		dec.Decode(&g)

		So(g[0].Game, ShouldResemble, backendtest.TestGame(backendtest.TestID))
		So(g[0].LocalizedName, ShouldEqual, backendtest.TestTerm)
	})

	Convey("should correct json for a game and locale", t, func() {
		params := getSearchParams(backendtest.TestTerm, "")
		params["locale"] = []string{backendtest.TestLocale}
		w := performRequest(params)

		So(w.Code, ShouldEqual, http.StatusOK)
		var g []*Game

		dec := json.NewDecoder(w.Body)
		dec.Decode(&g)

		So(g[0].Game, ShouldResemble, backendtest.TestGame(backendtest.TestID))
		So(g[0].Locale, ShouldEqual, backendtest.TestLocale)
		So(g[0].LocalizedName, ShouldEqual, backendtest.TestLocalizedName)
	})
}
