package api

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"net/url"
	"strconv"
	"testing"

	"code.justin.tv/web/discovery/backend/backendtest"
	"code.justin.tv/web/discovery/game"
	. "github.com/smartystreets/goconvey/convey"
)

func performGameRequest(params url.Values) *httptest.ResponseRecorder {
	be := backendtest.NewBackend()
	router := API{Backend: be}

	r, _ := http.NewRequest("GET", baseURL, nil)
	r.URL.RawQuery = params.Encode()

	return makeTestRequestWithContext(router.game, r)
}

func TestGame(t *testing.T) {
	Convey("should require one param", t, func() {
		params := url.Values{}
		w := performGameRequest(params)

		So(w.Code, ShouldEqual, http.StatusBadRequest)
		So(w.Body.String(), ShouldContainSubstring, "Requires 1 param")
	})

	Convey("should reject non int id arg", t, func() {
		params := url.Values{
			"id": []string{backendtest.TestBadTerm},
		}
		w := performGameRequest(params)

		So(w.Code, ShouldEqual, http.StatusBadRequest)
		So(w.Body.String(), ShouldContainSubstring, "Malformed id param")
	})

	Convey("should return a game for good id", t, func() {
		params := url.Values{
			"id": []string{strconv.Itoa(backendtest.TestID)},
		}
		w := performGameRequest(params)

		So(w.Code, ShouldEqual, http.StatusOK)
		So(w.Header().Get("Cache-Control"), ShouldEqual, fmt.Sprintf("public, max-age=%v", defaultMaxCacheAge))

		var g game.Game

		dec := json.NewDecoder(w.Body)
		dec.Decode(&g)

		So(g, ShouldResemble, backendtest.TestGame(backendtest.TestID))
	})

	Convey("should return a game for good name", t, func() {
		params := url.Values{
			"name": []string{backendtest.TestTerm},
		}
		w := performGameRequest(params)

		So(w.Code, ShouldEqual, http.StatusOK)
		So(w.Header().Get("Cache-Control"), ShouldEqual, fmt.Sprintf("public, max-age=%v", defaultMaxCacheAge))

		var g game.Game

		dec := json.NewDecoder(w.Body)
		dec.Decode(&g)

		So(g, ShouldResemble, backendtest.TestGame(backendtest.TestID))
	})

	Convey("should return a game for good alias", t, func() {
		params := url.Values{
			"name": []string{backendtest.TestAlias},
		}
		w := performGameRequest(params)

		So(w.Code, ShouldEqual, http.StatusOK)
		So(w.Header().Get("Cache-Control"), ShouldEqual, fmt.Sprintf("public, max-age=%v", defaultMaxCacheAge))

		var g game.Game

		dec := json.NewDecoder(w.Body)
		dec.Decode(&g)

		So(g, ShouldResemble, backendtest.TestGame(backendtest.TestID))
	})

	Convey("should return a game for good giantbomb id", t, func() {
		params := url.Values{
			"giantbomb_id": []string{strconv.Itoa(backendtest.TestGBID)},
		}
		w := performGameRequest(params)

		So(w.Code, ShouldEqual, http.StatusOK)
		So(w.Header().Get("Cache-Control"), ShouldEqual, fmt.Sprintf("public, max-age=%v", defaultMaxCacheAge))

		var g game.Game

		dec := json.NewDecoder(w.Body)
		dec.Decode(&g)

		So(g, ShouldResemble, backendtest.TestGame(backendtest.TestID))
	})

	Convey("should return not found for bad name", t, func() {
		params := url.Values{
			"name": []string{backendtest.TestBadTerm},
		}
		w := performGameRequest(params)

		So(w.Code, ShouldEqual, http.StatusNotFound)
	})

	Convey("should return not found for bad id", t, func() {
		params := url.Values{
			"id": []string{"322"},
		}
		w := performGameRequest(params)

		So(w.Code, ShouldEqual, http.StatusNotFound)
	})

	Convey("should return not found for bad giantbomb id", t, func() {
		params := url.Values{
			"giantbomb_id": []string{"322"},
		}
		w := performGameRequest(params)

		So(w.Code, ShouldEqual, http.StatusNotFound)
	})
}
