package api

import (
	"encoding/csv"
	"fmt"
	"net/http"
	"strconv"
	"strings"

	"code.justin.tv/common/chitin"
	"code.justin.tv/web/discovery/backend"
	"code.justin.tv/web/discovery/game"
	"github.com/jixwanwang/apiutils"
)

const totalHeader = "DSC-Collection-Total"
const defaultGameListLimit = 10

func (T *API) gamesList(w http.ResponseWriter, r *http.Request) {
	ctx := chitin.Context(w, r)

	params := r.URL.Query()

	locale := params.Get("locale")
	validLocale, errResponse := validateLocale(locale)
	if errResponse != nil {
		T.serveError(w, r, *errResponse)
		return
	}

	err := apiutils.RequireParams(params, []string{"id"})
	if err != nil {
		T.serveError(w, r, apiutils.NewErrorResponse(http.StatusBadRequest, err.Error()))
		return
	}

	stringIDs, err := csv.NewReader(strings.NewReader(params.Get("id"))).Read()
	if err != nil {
		T.serveError(w, r, apiutils.NewErrorResponse(http.StatusBadRequest, "Malformed id list, must be csv"))
		return
	}

	// Make a list of integer ids
	ids := []int{}
	for _, sID := range stringIDs {
		iID, err := strconv.Atoi(sID)
		if err != nil {
			T.serveError(w, r, apiutils.NewErrorResponse(http.StatusBadRequest, "Malformed id: "+sID))
			return
		}
		ids = append(ids, iID)
	}

	ordering := params.Get("ordering")
	if ordering != "name" {
		ordering = backend.DefaultOrdering
	}

	games, err := T.Backend.GameByIDs(ctx, ids, ordering)
	if err != nil {
		if err == backend.ErrGameNotFound {
			T.serveError(w, r, apiutils.NewErrorResponse(http.StatusNotFound, err.Error()))
			return
		}
		T.serveError(w, r, apiutils.NewErrorResponse(http.StatusInternalServerError, err.Error()))
		return
	}

	limit := defaultGameListLimit
	if len(params["limit"]) > 0 {
		lim, err := strconv.Atoi(params.Get("limit"))
		if err == nil {
			limit = lim
		}
	}

	// offset will just be 0 if it's missing or malformed
	offset, _ := strconv.Atoi(params.Get("offset"))

	w.Header().Add(totalHeader, strconv.Itoa(len(games)))

	paginatedGames := paginateGames(games, limit, offset)
	localizedGames, err := T.localizeGames(ctx, paginatedGames, validLocale)
	if err != nil {
		T.serveError(w, r, apiutils.NewErrorResponse(http.StatusInternalServerError, err.Error()))
		return
	}

	w.Header().Add("Cache-Control", fmt.Sprintf("public, max-age=%v", defaultMaxCacheAge))
	apiutils.ServeJSON(w, localizedGames)
}

func paginateGames(games []game.Game, limit, offset int) []game.Game {
	if offset > len(games) {
		return []game.Game{}
	} else if offset+limit >= len(games) {
		return games[offset:]
	} else {
		return games[offset : offset+limit]
	}
}
