package api

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"net/url"
	"strconv"
	"testing"

	"code.justin.tv/web/discovery/backend/backendtest"
	. "github.com/smartystreets/goconvey/convey"
)

func performGamesListRequest(params url.Values) *httptest.ResponseRecorder {
	be := backendtest.NewBackend()
	router := API{Backend: be}

	r, _ := http.NewRequest("GET", baseURL, nil)
	r.URL.RawQuery = params.Encode()

	return makeTestRequestWithContext(router.gamesList, r)
}

func TestGamesList(t *testing.T) {
	Convey("should require one param", t, func() {
		params := url.Values{}
		w := performGamesListRequest(params)

		So(w.Code, ShouldEqual, http.StatusBadRequest)
		So(w.Body.String(), ShouldContainSubstring, "Missing param: id")
	})

	Convey("should reject non int id arg", t, func() {
		params := url.Values{
			"id": []string{backendtest.TestBadTerm},
		}
		w := performGamesListRequest(params)

		So(w.Code, ShouldEqual, http.StatusBadRequest)
		So(w.Body.String(), ShouldContainSubstring, "Malformed id")
	})

	Convey("should return a game and localized name for good id", t, func() {
		params := url.Values{
			"id": []string{strconv.Itoa(backendtest.TestID)},
		}
		w := performGamesListRequest(params)

		So(w.Code, ShouldEqual, http.StatusOK)
		So(w.Header().Get("Cache-Control"), ShouldEqual, fmt.Sprintf("public, max-age=%v", defaultMaxCacheAge))

		var g []*Game

		dec := json.NewDecoder(w.Body)
		dec.Decode(&g)

		So(len(g), ShouldEqual, 1)
		So(g[0].Game, ShouldResemble, backendtest.TestGame(backendtest.TestID))
		So(g[0].LocalizedName, ShouldEqual, backendtest.TestTerm)
	})

	Convey("should return games and localized names for good id list", t, func() {
		params := url.Values{
			"id": []string{backendtest.TestIDsString},
		}
		w := performGamesListRequest(params)

		So(w.Code, ShouldEqual, http.StatusOK)
		So(w.Header().Get("Cache-Control"), ShouldEqual, fmt.Sprintf("public, max-age=%v", defaultMaxCacheAge))

		var g []*Game

		dec := json.NewDecoder(w.Body)
		dec.Decode(&g)

		for i, gg := range g {
			So(gg.ID, ShouldEqual, backendtest.TestIDs[i])
			So(gg.LocalizedName, ShouldEqual, backendtest.TestTerm)
		}
	})

	Convey("should return a game with localized name when locale present", t, func() {
		params := url.Values{
			"id":     []string{strconv.Itoa(backendtest.TestID)},
			"locale": []string{backendtest.TestLocale},
		}
		w := performGamesListRequest(params)

		So(w.Code, ShouldEqual, http.StatusOK)
		So(w.Header().Get("Cache-Control"), ShouldEqual, fmt.Sprintf("public, max-age=%v", defaultMaxCacheAge))

		var g []*Game

		dec := json.NewDecoder(w.Body)
		dec.Decode(&g)

		So(len(g), ShouldEqual, 1)
		So(g[0].Game, ShouldResemble, backendtest.TestGame(backendtest.TestID))
		So(g[0].Locale, ShouldEqual, backendtest.TestLocale)
		So(g[0].LocalizedName, ShouldEqual, backendtest.TestLocalizedName)
	})

	Convey("should return a game list with localized names when locale present", t, func() {
		params := url.Values{
			"id":     []string{backendtest.TestIDsString},
			"locale": []string{backendtest.TestLocale},
		}
		w := performGamesListRequest(params)

		So(w.Code, ShouldEqual, http.StatusOK)
		So(w.Header().Get("Cache-Control"), ShouldEqual, fmt.Sprintf("public, max-age=%v", defaultMaxCacheAge))

		var g []*Game

		dec := json.NewDecoder(w.Body)
		dec.Decode(&g)

		for i, gg := range g {
			So(gg.ID, ShouldEqual, backendtest.TestIDs[i])
			So(gg.Locale, ShouldEqual, backendtest.TestLocale)
			So(gg.LocalizedName, ShouldEqual, backendtest.TestLocalizedName)
		}
	})

	Convey("should cut off one game using limit", t, func() {
		params := url.Values{
			"id":    []string{backendtest.TestIDsString},
			"limit": []string{strconv.Itoa(len(backendtest.TestIDs) - 1)},
		}
		w := performGamesListRequest(params)

		So(w.Code, ShouldEqual, http.StatusOK)
		So(w.Header().Get("Cache-Control"), ShouldEqual, fmt.Sprintf("public, max-age=%v", defaultMaxCacheAge))

		var g []*Game

		dec := json.NewDecoder(w.Body)
		dec.Decode(&g)

		So(len(g), ShouldEqual, len(backendtest.TestIDs)-1)

		for i, gg := range g {
			So(gg.ID, ShouldEqual, backendtest.TestIDs[i])
		}
	})

	Convey("should correctly offset", t, func() {
		params := url.Values{
			"id":     []string{backendtest.TestIDsString},
			"offset": []string{strconv.Itoa(backendtest.TestOffset)},
		}
		w := performGamesListRequest(params)

		So(w.Code, ShouldEqual, http.StatusOK)
		So(w.Header().Get("Cache-Control"), ShouldEqual, fmt.Sprintf("public, max-age=%v", defaultMaxCacheAge))

		var g []*Game

		dec := json.NewDecoder(w.Body)
		dec.Decode(&g)

		So(len(g), ShouldEqual, len(backendtest.TestIDs)-backendtest.TestOffset)
		So(g[0].Game.ID, ShouldEqual, backendtest.TestIDs[backendtest.TestOffset])
	})

	Convey("should return no games for large offset", t, func() {
		params := url.Values{
			"id":     []string{backendtest.TestIDsString},
			"offset": []string{"123123123"},
		}
		w := performGamesListRequest(params)

		So(w.Code, ShouldEqual, http.StatusOK)
		So(w.Header().Get("Cache-Control"), ShouldEqual, fmt.Sprintf("public, max-age=%v", defaultMaxCacheAge))

		var g []*Game

		dec := json.NewDecoder(w.Body)
		dec.Decode(&g)

		So(len(g), ShouldEqual, 0)
	})

	Convey("should truncate list", t, func() {
		params := url.Values{
			"id":     []string{backendtest.TestIDsString},
			"limit":  []string{strconv.Itoa(backendtest.TestLimit)},
			"offset": []string{strconv.Itoa(len(backendtest.TestIDs) - backendtest.TestOffset)},
		}
		w := performGamesListRequest(params)

		So(w.Code, ShouldEqual, http.StatusOK)
		So(w.Header().Get("Cache-Control"), ShouldEqual, fmt.Sprintf("public, max-age=%v", defaultMaxCacheAge))

		var g []*Game

		dec := json.NewDecoder(w.Body)
		dec.Decode(&g)

		So(len(g), ShouldEqual, backendtest.TestOffset)
		So(g[len(g)-1].Game.ID, ShouldEqual, backendtest.TestIDs[len(backendtest.TestIDs)-1])
	})
}
