package api

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"net/url"
	"strconv"
	"testing"

	"code.justin.tv/web/discovery/backend/backendtest"
	. "github.com/smartystreets/goconvey/convey"
)

func performGetAllAliasesRequest(params url.Values) *httptest.ResponseRecorder {
	be := backendtest.NewBackend()
	router := API{Backend: be}

	r, _ := http.NewRequest("GET", baseURL, nil)
	r.URL.RawQuery = params.Encode()

	return makeTestRequestWithContext(router.getAllAliases, r)
}

func TestGetAllAliases(t *testing.T) {
	Convey("should require one param", t, func() {
		params := url.Values{}
		w := performGetAllAliasesRequest(params)

		So(w.Code, ShouldEqual, http.StatusBadRequest)
	})

	Convey("should reject non int id arg", t, func() {
		params := url.Values{
			"id": []string{backendtest.TestBadTerm},
		}
		w := performGetAllAliasesRequest(params)

		So(w.Code, ShouldEqual, http.StatusBadRequest)
		So(w.Body.String(), ShouldContainSubstring, "Malformed id")
	})

	Convey("should return aliases for good id", t, func() {
		params := url.Values{
			"id": []string{strconv.Itoa(backendtest.TestID)},
		}
		w := performGetAllAliasesRequest(params)

		So(w.Code, ShouldEqual, http.StatusOK)
		So(w.Header().Get("Cache-Control"), ShouldEqual, fmt.Sprintf("public, max-age=%v", defaultMaxCacheAge))

		var a aliasesResponse

		dec := json.NewDecoder(w.Body)
		dec.Decode(&a)

		So(len(a.Aliases), ShouldEqual, 1)
		So(a.Aliases[0], ShouldResemble, backendtest.TestAlias)
	})

	Convey("should return a game for good name", t, func() {
		params := url.Values{
			"name": []string{backendtest.TestTerm},
		}
		w := performGetAllAliasesRequest(params)

		So(w.Code, ShouldEqual, http.StatusOK)
		So(w.Header().Get("Cache-Control"), ShouldEqual, fmt.Sprintf("public, max-age=%v", defaultMaxCacheAge))

		var a aliasesResponse

		dec := json.NewDecoder(w.Body)
		dec.Decode(&a)

		So(len(a.Aliases), ShouldEqual, 1)
		So(a.Aliases[0], ShouldResemble, backendtest.TestAlias)
	})
}
