package api

import (
	"context"
	"errors"
	"fmt"
	"net/http"
	"strconv"

	"code.justin.tv/web/discovery/backend"
	"code.justin.tv/web/discovery/game"
	"github.com/jixwanwang/apiutils"
)

type Game struct {
	game.Game
	Locale        string `json:"locale"`
	LocalizedName string `json:"localized_name"`
}

func (T *API) getGame(ctx context.Context, w http.ResponseWriter, r *http.Request) (*Game, error) {
	params := r.URL.Query()

	var (
		id  int
		err error
		g   game.Game
	)

	locale := params.Get("locale")
	validLocale, errResponse := validateLocale(locale)
	if errResponse != nil {
		T.serveError(w, r, *errResponse)
		return nil, errResponse
	}

	if len(params["name"]) > 0 {
		g, err = T.Backend.GameByName(ctx, params.Get("name"))
		if err == backend.ErrGameNotFound {
			g, err = T.Backend.GameByAlias(ctx, params.Get("name"))
		}
	} else if len(params["id"]) > 0 {
		id, err = strconv.Atoi(params.Get("id"))
		if err != nil {
			T.serveError(w, r, apiutils.NewErrorResponse(http.StatusBadRequest, "Malformed id parameter"))
			return nil, err
		}

		g, err = T.Backend.GameByID(ctx, id)
	} else if len(params["giantbomb_id"]) > 0 {
		id, err = strconv.Atoi(params.Get("giantbomb_id"))
		if err != nil {
			T.serveError(w, r, apiutils.NewErrorResponse(
				http.StatusBadRequest,
				fmt.Sprintf("Malformed giantbomb_id parameter: %s", params.Get("giantbomb_id")),
			))
			return nil, err
		}

		g, err = T.Backend.GameByGBID(ctx, id)
	} else {
		T.serveError(w, r, apiutils.NewErrorResponse(http.StatusBadRequest, "Requires 1 param of name, id, giantbomb_id"))
		return nil, errors.New("Requires 1 param of name, id, giantbomb_id")
	}

	if err != nil {
		if err == backend.ErrGameNotFound {
			T.serveError(w, r, apiutils.NewErrorResponse(http.StatusNotFound, err.Error()))
			return nil, err
		}
		T.serveError(w, r, apiutils.NewErrorResponse(http.StatusInternalServerError, err.Error()))
		return nil, err
	}

	games, err := T.localizeGames(ctx, []game.Game{g}, validLocale)
	if err != nil {
		T.serveError(w, r, apiutils.NewErrorResponse(http.StatusInternalServerError, err.Error()))
		return nil, err
	}

	return games[0], nil
}
