package api

import (
	"net/http"
	"net/http/httptest"
	"net/url"
	"strconv"
	"testing"

	"code.justin.tv/common/chitin"
	"code.justin.tv/web/discovery/backend/backendtest"
	. "github.com/smartystreets/goconvey/convey"
)

func getGame(params url.Values) (*httptest.ResponseRecorder, *Game, error) {
	be := backendtest.NewBackend()
	router := API{Backend: be}

	r, _ := http.NewRequest("GET", baseURL, nil)
	r.URL.RawQuery = params.Encode()
	w := httptest.NewRecorder()

	var g *Game
	var err error

	handler := chitin.Handler(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		g, err = router.getGame(chitin.Context(w, r), w, r)
	}))
	handler.ServeHTTP(w, r)

	return w, g, err
}

func TestGetGame(t *testing.T) {
	Convey("should require one param", t, func() {
		params := url.Values{}
		w, _, err := getGame(params)

		So(w.Code, ShouldEqual, http.StatusBadRequest)
		So(w.Body.String(), ShouldContainSubstring, "Requires 1 param")
		So(err, ShouldNotBeNil)
	})

	Convey("should reject non int id arg", t, func() {
		params := url.Values{
			"id": []string{backendtest.TestBadTerm},
		}
		w, _, err := getGame(params)

		So(w.Code, ShouldEqual, http.StatusBadRequest)
		So(w.Body.String(), ShouldContainSubstring, "Malformed id param")
		So(err, ShouldNotBeNil)
	})

	Convey("should return a game and localized name for good id", t, func() {
		params := url.Values{
			"id": []string{strconv.Itoa(backendtest.TestID)},
		}
		_, g, err := getGame(params)

		So(g.Game, ShouldResemble, backendtest.TestGame(backendtest.TestID))
		So(g.LocalizedName, ShouldEqual, backendtest.TestTerm)
		So(err, ShouldBeNil)
	})

	Convey("should return a game and localized name for good name", t, func() {
		params := url.Values{
			"name": []string{backendtest.TestTerm},
		}
		_, g, err := getGame(params)

		So(g.Game, ShouldResemble, backendtest.TestGame(backendtest.TestID))
		So(g.LocalizedName, ShouldEqual, backendtest.TestTerm)
		So(err, ShouldBeNil)
	})

	Convey("should return a game and localized name for good alias", t, func() {
		params := url.Values{
			"name": []string{backendtest.TestAlias},
		}
		_, g, err := getGame(params)

		So(g.Game, ShouldResemble, backendtest.TestGame(backendtest.TestID))
		So(g.LocalizedName, ShouldEqual, backendtest.TestTerm)
		So(err, ShouldBeNil)
	})

	Convey("should return a game and localized name for good name (match on game name over game alias)", t, func() {
		// TestTerm3 (Game3's name) == TestAlias2 (Game2's alias)
		params := url.Values{
			"name": []string{backendtest.TestTerm3},
		}
		_, g, err := getGame(params)

		So(g.Game, ShouldResemble, backendtest.TestGame(backendtest.TestID3))
		So(g.LocalizedName, ShouldEqual, backendtest.TestTerm3)
		So(err, ShouldBeNil)
	})

	Convey("should return a game and localized name for good giantbomb id", t, func() {
		params := url.Values{
			"giantbomb_id": []string{strconv.Itoa(backendtest.TestGBID)},
		}
		_, g, err := getGame(params)

		So(g.Game, ShouldResemble, backendtest.TestGame(backendtest.TestID))
		So(g.LocalizedName, ShouldEqual, backendtest.TestTerm)
		So(err, ShouldBeNil)
	})

	Convey("should return a game for good id and locale", t, func() {
		params := url.Values{
			"id":     []string{strconv.Itoa(backendtest.TestID)},
			"locale": []string{backendtest.TestLocale},
		}
		_, g, err := getGame(params)

		So(g.Game, ShouldResemble, backendtest.TestGame(backendtest.TestID))
		So(g.Locale, ShouldEqual, backendtest.TestLocale)
		So(g.LocalizedName, ShouldEqual, backendtest.TestLocalizedName)
		So(err, ShouldBeNil)
	})

	Convey("should return not found for bad name", t, func() {
		params := url.Values{
			"name": []string{backendtest.TestBadTerm},
		}
		w, _, err := getGame(params)

		So(w.Code, ShouldEqual, http.StatusNotFound)
		So(err, ShouldNotBeNil)
	})

	Convey("should return not found for bad id", t, func() {
		params := url.Values{
			"id": []string{"322"},
		}
		w, _, err := getGame(params)

		So(w.Code, ShouldEqual, http.StatusNotFound)
		So(err, ShouldNotBeNil)
	})

	Convey("should return not found for bad giantbomb id", t, func() {
		params := url.Values{
			"giantbomb_id": []string{"322"},
		}
		w, _, err := getGame(params)

		So(w.Code, ShouldEqual, http.StatusNotFound)
		So(err, ShouldNotBeNil)
	})
}
